import React, { useEffect, useState } from 'react';
import {
    Table,
    Button,
    Modal,
    Image,
    message,
    Tag,
    Space,
    Spin,
    Tooltip,
} from 'antd';
import { ExclamationCircleOutlined } from '@ant-design/icons';
import {
    getAllRequests,
    deleteRequest,
} from '../api/request';
import { useNavigate } from 'react-router-dom';

const { confirm } = Modal;

const RequestAdminPanel = () => {
    const [requests, setRequests] = useState([]);
    const [loading, setLoading] = useState(false);
    const navigate = useNavigate();

    // 获取所有求助帖
    const fetchRequests = async () => {
        setLoading(true);
        try {
            const data = await getAllRequests();
            setRequests(data);
        } catch (error) {
            message.error('获取求助帖失败');
        } finally {
            setLoading(false);
        }
    };

    useEffect(() => {
        fetchRequests();
    }, []);

    // 删除确认弹窗
    const showDeleteConfirm = (requestid) => {
        confirm({
            title: '确认删除该求助帖吗？',
            icon: <ExclamationCircleOutlined />,
            okText: '删除',
            okType: 'danger',
            cancelText: '取消',
            onOk() {
                handleDelete(requestid);
            },
        });
    };

    // 删除求助帖
    const handleDelete = async (requestid) => {
        try {
            const success = await deleteRequest(requestid);
            if (success) {
                message.success('删除成功');
                fetchRequests();
            } else {
                message.error('删除失败');
            }
        } catch {
            message.error('删除请求失败');
        }
    };

    // 处理按钮示例
    const handleProcess = (request) => {
        navigate(`/process/${request.requestid}`, {
            state: {
                requestid: request.requestid,
                helpedid: request.userid,
                loaduser: request.loaduser,
                money: request.money,
                torrentid: request.torrentid
            },
        });
    };

    console.log('请求数据:', requests);

    // 表格列定义
    const columns = [
        {
            title: 'ID',
            dataIndex: 'requestid',
            key: 'requestid',
            width: 60,
            fixed: 'left',
        },
        {
            title: '发帖ID',
            dataIndex: 'userid',
            key: 'userid',
            width: 100,
        },
        {
            title: '上传者ID',
            dataIndex: 'loaduser',
            key: 'loaduser',
            width: 100,
            render: (val) => val ?? <Tag color="default">目前没有</Tag>,
        },
        {
            title: '资源名字',
            dataIndex: 'name',
            key: 'name',
            width: 120,
            ellipsis: true,
        },
        {
            title: '内容描述',
            dataIndex: 'plot',
            key: 'plot',
            ellipsis: { showTitle: false },
            render: (text) => (
                <Tooltip placement="topLeft" title={text}>
                    {text}
                </Tooltip>
            ),
        },
        {
            title: '悬赏金额',
            dataIndex: 'money',
            key: 'money',
            width: 80,
            render: (money) => <Tag color="volcano">{money}元</Tag>,
        },
        {
            title: '照片',
            dataIndex: 'photo',
            key: 'photo',
            width: 100,
            render: (url) =>
                url ? (
                    <Image
                        src={`http://localhost:8080${url}`}
                        width={80}
                        height={80}
                        style={{ objectFit: 'cover' }}
                        placeholder={<Spin />}
                    />
                ) : (
                    <Tag color="default">无</Tag>
                ),
        },
        {
            title: '年份',
            dataIndex: 'year',
            key: 'year',
            width: 80,
        },
        {
            title: '国家',
            dataIndex: 'country',
            key: 'country',
            width: 100,
        },
        {
            title: '种子号',
            dataIndex: 'torrentid',
            key: 'torrentid',
            width: 120,
            render: (val) =>
                val !== null && val !== undefined ? (
                    <Tag color="green">{val}</Tag>
                ) : (
                    <Tag color="default">暂无</Tag>
                ),
        },
        {
            title: '发布时间',
            dataIndex: 'requesttime',
            key: 'requesttime',
            width: 180,
            render: (text) => {
                if (!text) return <Tag color="default">暂无</Tag>;
                return new Date(text).toLocaleString();
            },
        },
        {
            title: '操作',
            key: 'action',
            fixed: 'right',
            width: 150,
            render: (_, record) => (
                <Space size="middle">
                    <Button
                        type="primary"
                        onClick={() => handleProcess(record)}
                    >
                        处理
                    </Button>
                    <Button
                        danger
                        onClick={() => showDeleteConfirm(record.requestid)}
                    >
                        删除
                    </Button>
                </Space>
            ),
        },
    ];

    return (
        <div style={{ padding: 20 }}>
            <h2 style={{ marginBottom: 20 }}>求助帖管理面板</h2>
            <Table
                rowKey="requestid"
                columns={columns}
                dataSource={requests}
                loading={loading}
                scroll={{ x: 1600 }}
                pagination={{ pageSize: 10 }}
                bordered
                size="middle"
            />
        </div>
    );
};

export default RequestAdminPanel;
