import React, { useState, useEffect } from 'react';
import axios from 'axios';
import './AuthStyle.css'; // 引入样式文件

const UserAuth = () => {
  // 状态管理
  const [registerData, setRegisterData] = useState({
    username: '',
    password: '',
    code: '',
    email: '',
    emailcode: ''
  });
  const [loginData, setLoginData] = useState({
    username: '',
    password: ''
  });
  const [activeTab, setActiveTab] = useState('login');
  const [message, setMessage] = useState('');
  const [error, setError] = useState('');
  const [countdown, setCountdown] = useState(0);
  const [isLoading, setIsLoading] = useState(false);

  // 倒计时效果
  useEffect(() => {
    if (countdown > 0) {
      const timer = setTimeout(() => setCountdown(countdown - 1), 1000);
      return () => clearTimeout(timer);
    }
  }, [countdown]);

  // 输入处理
  const handleChange = (e, formType) => {
    const { name, value } = e.target;
    formType === 'register' 
      ? setRegisterData(prev => ({ ...prev, [name]: value }))
      : setLoginData(prev => ({ ...prev, [name]: value }));
  };

  // 发送验证码
  const handleSendCode = async () => {
    if (!registerData.email) {
      setError('请输入邮箱地址');
      return;
    }
    
    try {
      setIsLoading(true);
      const response = await axios.post('http://localhost:8080/user/sendCode', null, {
        params: { email: registerData.email }
      });
      
      if (response.data?.success) {
        setMessage('验证码已发送');
        setError('');
        setCountdown(60);
      } else {
        setError(response.data?.message || '发送失败');
      }
    } catch (err) {
      setError(err.response?.data?.message || '发送出错');
    } finally {
      setIsLoading(false);
    }
  };

  // 注册提交
  const handleRegister = async (e) => {
    e.preventDefault();
    if (Object.values(registerData).some(v => !v)) {
      setError('请填写所有字段');
      return;
    }
    
    try {
      setIsLoading(true);
      const response = await axios.post('http://localhost:8080/user/register', {
        username: registerData.username,
        password: registerData.password,
        email: registerData.email
      }, {
        params: {
          code: registerData.code,
          emailcode: registerData.emailcode
        }
      });
      
      if (response.data?.success) {
        setMessage('注册成功');
        setActiveTab('login');
        setRegisterData({
          username: '',
          password: '',
          code: '',
          email: '',
          emailcode: ''
        });
      } else {
        setError(response.data?.message || '注册失败');
      }
    } catch (err) {
      setError(err.response?.data?.message || '注册出错');
    } finally {
      setIsLoading(false);
    }
  };

  // 登录提交
  const handleLogin = async (e) => {
    e.preventDefault();
    if (!loginData.username || !loginData.password) {
      setError('请输入用户名和密码');
      return;
    }
    
    try {
      setIsLoading(true);
      const response = await axios.post('http://localhost:8080/user/login', null, {
        params: {
          username: loginData.username,
          password: loginData.password
        }
      });
      
      if (response.data?.success) {
        localStorage.setItem('user', JSON.stringify(response.data.user));
        if(response.data.user.permission==0)
        window.location.href = '/';
        else
        window.location.href = '/';
      } else {
        setError(response.data?.message || '登录失败');
      }
    } catch (err) {
      setError(err.response?.data?.message || '登录出错');
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <div className="auth-wrapper">
    <div className="auth-container">
      <div className="auth-header">
        <h1>PT Share</h1>
        <p>专业资源分享平台</p>
      </div>
      
      <div className="auth-card">
        <div className="tabs">
          <button 
            className={`tab-btn ${activeTab === 'login' ? 'active' : ''}`}
            onClick={() => setActiveTab('login')}
          >
            登录
          </button>
          <button 
            className={`tab-btn ${activeTab === 'register' ? 'active' : ''}`}
            onClick={() => setActiveTab('register')}
          >
            注册
          </button>
          <div 
            className="tab-indicator" 
            style={{ left: activeTab === 'login' ? '0' : '50%' }}
          />
        </div>
        
        <div className={`message ${error ? 'error' : ''}`}>
          {error || message}
        </div>
        
        {activeTab === 'login' ? (
          <form onSubmit={handleLogin} className="auth-form">
            <div className="form-group">
              <input
                type="text"
                name="username"
                value={loginData.username}
                onChange={(e) => handleChange(e, 'login')}
                placeholder="用户名"
                required
              />
            </div>
            <div className="form-group">
              <input
                type="password"
                name="password"
                value={loginData.password}
                onChange={(e) => handleChange(e, 'login')}
                placeholder="密码"
                required
              />
            </div>
            <button type="submit" className="submit-btn" disabled={isLoading}>
              {isLoading ? <div className="spinner"></div> : '登 录'}
            </button>
          </form>
        ) : (
          <form onSubmit={handleRegister} className="auth-form">
            <div className="form-group">
              <input
                type="text"
                name="username"
                value={registerData.username}
                onChange={(e) => handleChange(e, 'register')}
                placeholder="用户名"
                required
              />
            </div>
            <div className="form-group">
              <input
                type="password"
                name="password"
                value={registerData.password}
                onChange={(e) => handleChange(e, 'register')}
                placeholder="密码"
                required
              />
            </div>
            <div className="form-group">
              <input
                type="text"
                name="code"
                value={registerData.code}
                onChange={(e) => handleChange(e, 'register')}
                placeholder="邀请码"
                required
              />
            </div>
            <div className="form-group">
              <input
                type="email"
                name="email"
                value={registerData.email}
                onChange={(e) => handleChange(e, 'register')}
                placeholder="邮箱"
                required
              />
            </div>
            <div className="form-group with-button">
              <input
                type="text"
                name="emailcode"
                value={registerData.emailcode}
                onChange={(e) => handleChange(e, 'register')}
                placeholder="验证码"
                required
              />
              <button 
                type="button" 
                onClick={handleSendCode}
                disabled={countdown > 0 || isLoading}
                className="code-btn"
              >
                {countdown > 0 ? `${countdown}s` : '获取验证码'}
              </button>
            </div>
            <button type="submit" className="submit-btn" disabled={isLoading}>
              {isLoading ? <div className="spinner"></div> : '注 册'}
            </button>
          </form>
        )}
      </div>
      
      <div className="auth-footer">
        <p>© {new Date().getFullYear()} PT Share - 专业资源分享平台</p>
      </div>
    </div>
    </div>
  );
};

export default UserAuth;