import React, { useState, useEffect } from 'react';
import { addFriend, deleteFriend, getFriendsByUserId } from '../api/friends';

const FriendManager = ({ currentUser, onSelectRelation }) => {
    const [friendId, setFriendId] = useState('');
    const [friends, setFriends] = useState([]);
    const [isLoading, setIsLoading] = useState(false);
    const [isRefreshing, setIsRefreshing] = useState(false);

    useEffect(() => {
        if (currentUser?.id) loadFriends(currentUser.id);
    }, [currentUser]);

    const loadFriends = async (userid) => {
        setIsRefreshing(true);
        try {
            const res = await getFriendsByUserId(userid);
            setFriends(res.data);
        } catch (err) {
            console.error('加载好友失败', err);
            alert('加载好友失败，请稍后重试');
        }
        setIsRefreshing(false);
    };

    const handleFriendIdChange = (e) => {
        const value = e.target.value;
        if (/^\d*$/.test(value)) setFriendId(value);
    };

    const handleAddFriend = async () => {
        if (!friendId) return alert('请输入好友ID');

        const newFriendId = parseInt(friendId, 10);
        if (newFriendId === currentUser.id) return alert('不能添加自己为好友');

        if (friends.some(f => f.friend1 === newFriendId || f.friend2 === newFriendId)) {
            return alert('该用户已是您的好友');
        }

        setIsLoading(true);
        try {
            const res = await addFriend({ friend1: currentUser.id, friend2: newFriendId });
            if (res.data) {
                alert('添加成功');
                setFriendId('');
                loadFriends(currentUser.id);
            } else {
                alert('添加失败');
            }
        } catch (err) {
            alert('添加好友失败');
            console.error(err);
        }
        setIsLoading(false);
    };

    /* ---------- 这里开始：删除好友逻辑改为 friend1 + friend2 ---------- */
    const handleDelete = async (friend1, friend2) => {
        if (!window.confirm('确认删除该好友吗？')) return;
        setIsLoading(true);
        try {
            const res = await deleteFriend(friend1, friend2);
            if (res.data) {
                alert('删除成功');
                loadFriends(currentUser.id);
            } else {
                alert('删除失败');
            }
        } catch (err) {
            alert('删除好友失败');
            console.error(err);
        }
        setIsLoading(false);
    };
    /* ------------------------------------------------------------------- */

    return (
        <div className="max-w-xl mx-auto p-4">
            <h2 className="text-2xl font-bold mb-4">好友管理</h2>

            {/* 添加好友区域 */}
            <div className="mb-6 space-y-2">
                <input
                    type="text"
                    placeholder="输入好友的用户ID"
                    value={friendId}
                    onChange={handleFriendIdChange}
                    className="border p-2 rounded w-full"
                />
                <button
                    onClick={handleAddFriend}
                    disabled={isLoading}
                    className={`bg-blue-500 text-white px-4 py-2 rounded hover:bg-blue-600 ${isLoading ? 'opacity-50 cursor-not-allowed' : ''}`}
                >
                    {isLoading ? '添加中…' : '添加好友'}
                </button>
            </div>

            {/* 好友列表 */}
            <div className="flex justify-between items-center mb-3">
                <h3 className="text-xl font-semibold">我的好友列表</h3>
                <button
                    onClick={() => loadFriends(currentUser.id)}
                    disabled={isRefreshing}
                    className="text-sm text-blue-500 hover:underline disabled:text-gray-400"
                >
                    {isRefreshing ? '刷新中…' : '刷新'}
                </button>
            </div>

            {friends.length === 0 ? (
                <p className="text-gray-500">暂无好友</p>
            ) : (
                <ul className="space-y-3">
                    {friends.map((f) => {
                        const friendUserId =
                            f.friend1 === currentUser.id ? f.friend2 : f.friend1;
                        return (
                            <li
                                key={f.relationid}
                                className="border p-3 rounded flex justify-between items-center hover:bg-gray-100 cursor-pointer"
                                onClick={() =>
                                    onSelectRelation({
                                        relationid: f.relationid,
                                        friendId: friendUserId,
                                    })
                                }
                            >
                                <div>
                                    <p>好友用户ID：{friendUserId}</p>
                                    <p className="text-sm text-gray-500">
                                        添加时间：{new Date(f.requestTime).toLocaleString()}
                                    </p>
                                </div>
                                <button
                                    onClick={(e) => {
                                        e.stopPropagation();
                                        /* ------- 传入正确的 friend1 & friend2 -------- */
                                        handleDelete(f.friend1, f.friend2);
                                    }}
                                    className="text-red-500 hover:underline"
                                    disabled={isLoading}
                                >
                                    删除
                                </button>
                            </li>
                        );
                    })}
                </ul>
            )}
        </div>
    );
};

export default FriendManager;
