import { createRouter, createWebHistory } from 'vue-router'
import { ElMessage } from 'element-plus'
import store from '@/store'

// 路由组件
import LoginView from '@/views/auth/LoginView.vue'
import RegisterView from '@/views/auth/RegisterView.vue'
import HomeView from '@/views/HomeView.vue'

const routes = [
  {
    path: '/',
    redirect: '/login'
  },
  {
    path: '/login',
    name: 'Login',
    component: LoginView,
    meta: {
      title: 'PT Tracker - 登录',
      requiresGuest: true
    }
  },
  {
    path: '/register',
    name: 'Register',
    component: RegisterView,
    meta: {
      title: 'PT Tracker - 注册',
      requiresGuest: true
    }
  },
  {
    path: '/home',
    name: 'Home',
    component: HomeView,
    meta: {
      title: 'PT Tracker - 首页',
      requiresAuth: true
    }
  },
  {
    path: '/profile',
    name: 'Profile',
    component: () => import('@/views/auth/ProfileView.vue'),
    meta: {
      title: 'PT Tracker - 个人资料',
      requiresAuth: true
    }
  },
  // 种子相关路由
  {
    path: '/torrents',
    name: 'Torrents',
    component: () => import('@/views/torrent/TorrentsView.vue'),
    meta: {
      title: 'PT Tracker - 种子浏览',
      requiresAuth: true
    }
  },
  {
    path: '/upload',
    name: 'Upload',
    component: () => import('@/views/torrent/UploadView.vue'),
    meta: {
      title: 'PT Tracker - 上传种子',
      requiresAuth: true
    }
  },
  {
    path: '/torrent/:id',
    name: 'TorrentDetail',
    component: () => import('@/views/torrent/TorrentDetailView.vue'),
    meta: {
      title: 'PT Tracker - 种子详情',
      requiresAuth: true
    }
  },
  // 论坛相关路由
  {
    path: '/forum',
    name: 'Forum',
    component: () => import('@/views/forum/ForumView.vue'),
    meta: {
      title: 'PT Tracker - 论坛',
      requiresAuth: true
    }
  },
  {
    path: '/forum/section/:id',
    name: 'ForumSection',
    component: () => import('@/views/forum/ForumSectionView.vue'),
    meta: {
      title: 'PT Tracker - 版块',
      requiresAuth: true
    }
  },
  {
    path: '/forum/topic/:id',
    name: 'ForumTopic',
    component: () => import('@/views/forum/ForumTopicView.vue'),
    meta: {
      title: 'PT Tracker - 主题详情',
      requiresAuth: true
    }
  },
  // 404页面
  {
    path: '/:pathMatch(.*)*',
    redirect: '/login'
  }
]

const router = createRouter({
  history: createWebHistory(),
  routes
})

// 更新路由守卫
router.beforeEach(async (to, from, next) => {
  // 设置页面标题
  if (to.meta.title) {
    document.title = to.meta.title
  }
  
  console.log(`路由跳转: ${from.path} -> ${to.path}`)
  
  // 恢复登录状态（仅在应用启动时执行一次）
  if (!store.state.auth.isLoggedIn && localStorage.getItem('isLoggedIn') === 'true') {
    store.dispatch('auth/restoreLoginState')
  }
  
  const isLoggedIn = store.getters['auth/isAuthenticated']
  
  // 需要登录但未登录
  if (to.meta.requiresAuth && !isLoggedIn) {
    // 尝试检查服务器端的登录状态
    try {
      const isValid = await store.dispatch('auth/checkLoginStatus')
      if (isValid) {
        // 服务器确认已登录，继续跳转
        next()
      } else {
        // 服务器确认未登录，跳转到登录页
        ElMessage.warning('请先登录')
        next('/login')
      }
    } catch (error) {
      // 检查失败，跳转到登录页
      ElMessage.warning('请先登录')
      next('/login')
    }
    return
  }
  
  // 已登录但访问登录/注册页
  if (to.meta.requiresGuest && isLoggedIn) {
    next('/home')
    return
  }
  
  next()
})

export default router