import request from './request'

/**
 * 上传种子
 * @param {FormData} formData - 包含种子文件和相关信息的表单数据
 * @returns {Promise}
 */
export function uploadTorrent(formData) {
  return request({
    url: '/torrent/upload',
    method: 'post',
    data: formData,
    headers: {
      'Content-Type': 'multipart/form-data'
    }
  })
}

/**
 * 搜索种子
 * @param {Object} searchParams - 搜索参数
 * @param {string} searchParams.keyword - 搜索关键词
 * @param {string} searchParams.category - 分类

 * @param {string} searchParams.sortBy - 排序字段
 * @param {string} searchParams.sortOrder - 排序方向
 * @param {number} searchParams.page - 页码
 * @param {number} searchParams.entriesPerPage - 每页条数
 * @returns {Promise}
 */
export function searchTorrents(searchParams) {
  const hasCategory = searchParams.category 
  
  if (hasCategory) {
    // 使用GET方式，支持category查询参数
    const queryParams = new URLSearchParams()
    
    if (searchParams.keyword) {
      queryParams.append('keyword', searchParams.keyword)
    }
    queryParams.append('entriesPerPage', searchParams.entriesPerPage || 20)
    queryParams.append('page', searchParams.page || 0)
    
    // 优先使用category参数，如果没有则使用categoryId
    const categoryValue = searchParams.category || searchParams.categoryId
    if (categoryValue) {
      queryParams.append('category', categoryValue)
    }
    
    const url = `/torrent/search?${queryParams.toString()}`
    console.log('🔍 使用GET方式搜索种子，URL:', url)
    
    return request({
      url: url,
      method: 'get'
    }).catch(error => {
      console.error('🚨 GET API请求失败:', {
        url: url,
        requestedURL: error.config?.url,
        baseURL: error.config?.baseURL,
        fullURL: error.config?.baseURL ? error.config.baseURL + error.config.url : error.config?.url,
        status: error.response?.status,
        statusText: error.response?.statusText
      })
      throw error
    })
  } else {
    // 使用POST方式，兼容原有的搜索方式
    console.log('🔍 使用POST方式搜索种子，参数:', searchParams)
    
    return request({
      url: '/torrent/search',
      method: 'post',
      data: searchParams
    }).catch(error => {
      console.error('🚨 POST API请求失败:', {
        url: '/torrent/search',
        requestedURL: error.config?.url,
        baseURL: error.config?.baseURL,
        fullURL: error.config?.baseURL ? error.config.baseURL + error.config.url : error.config?.url,
        status: error.response?.status,
        statusText: error.response?.statusText
      })
      throw error
    })
  }
}

/**
 * 获取分类列表
 * @returns {Promise}
 */
export function getCategories() {
  return request({
    url: '/category/list', // 注意这里不需要加 /api，已经在代理中配置了
    method: 'get'
  })
}

/**
 * 获取标签列表
 * @returns {Promise}
 * 暂时还没有获取标签的列表
 */
export function getTags() {
    console.log('调用获取标签列表API...')
    // 后端没有标签的Controller，返回模拟数据
    return new Promise((resolve) => {
      setTimeout(() => {
        resolve({
          data: [
            { id: 1, name: 'linux' },
            { id: 2, name: 'ios' },
            { id: 3, name: 'ubuntu' }
          ]
        })
      }, 100)
    })
  }

/**
 * 获取种子详情
 * @param {string} infoHash - 种子的info hash
 * @returns {Promise}
 */
export function getTorrentDetail(infoHash) {
  return request({
    url: `/torrent/view/${infoHash}`,
    method: 'get'
  })
}

/**
 * 下载种子
 * @param {string} infoHash - 种子的info hash
 * @returns {Promise}
 */
export function downloadTorrent(infoHash) {
  return request({
    url: `/torrent/download/${infoHash}`,
    method: 'get'
  })
}