import request from './request'

export const userApi = {
  /**
   * 获取当前用户信息
   * @returns {Promise<Object>} 用户信息
   */
  getCurrentUser() {
    return request({
      url: '/user/profile',
      method: 'get'
    })
  },

  /**
   * 更新用户资料
   * @param {Object} profileData - 用户资料数据
   * @returns {Promise<Object>} 更新响应
   */
  updateProfile(profileData) {
    return request({
      url: '/user/profile',
      method: 'put',
      data: profileData
    })
  },

  /**
   * 修改密码
   * @param {Object} passwordData - 密码数据
   * @param {string} passwordData.currentPassword - 当前密码
   * @param {string} passwordData.newPassword - 新密码
   * @returns {Promise<Object>} 修改响应
   */
  changePassword(passwordData) {
    return request({
      url: '/user/password',
      method: 'put',
      data: passwordData
    })
  },

  /**
   * 上传头像
   * @param {FormData} formData - 包含头像文件的FormData
   * @returns {Promise<Object>} 上传响应
   */
  uploadAvatar(formData) {
    return request({
      url: '/user/avatar',
      method: 'post',
      data: formData,
      headers: {
        'Content-Type': 'multipart/form-data'
      }
    })
  },

  /**
   * 获取用户统计信息
   * @returns {Promise<Object>} 统计信息
   */
  getUserStats() {
    return request({
      url: '/user/stats',
      method: 'get'
    })
  },

  /**
   * 获取用户活动记录
   * @param {Object} params - 查询参数
   * @param {string} params.type - 活动类型
   * @param {number} params.page - 页码
   * @param {number} params.size - 每页大小
   * @returns {Promise<Object>} 活动记录
   */
  getUserActivities(params = {}) {
    return request({
      url: '/user/activities',
      method: 'get',
      params
    })
  },

  /**
   * 获取用户上传的种子
   * @param {Object} params - 查询参数
   * @param {number} params.page - 页码
   * @param {number} params.size - 每页大小
   * @returns {Promise<Object>} 种子列表
   */
  getUserTorrents(params = {}) {
    return request({
      url: '/user/torrents',
      method: 'get',
      params
    })
  },

  /**
   * 获取登录历史
   * @param {Object} params - 查询参数
   * @param {number} params.page - 页码
   * @param {number} params.size - 每页大小
   * @returns {Promise<Object>} 登录历史
   */
  getLoginHistory(params = {}) {
    return request({
      url: '/user/login-history',
      method: 'get',
      params
    })
  }
}
