package com.github.example.pt.controller;

import com.github.example.pt.entity.Post;
import com.github.example.pt.entity.User;
import com.github.example.pt.service.PostLikeService;
import com.github.example.pt.service.PostService;
import com.github.example.pt.service.UserService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.HashMap;
import java.util.Map;

@RestController
@RequestMapping("/api/post-likes")
public class PostLikeController {

    @Autowired
    private PostLikeService postLikeService;

    @Autowired
    private PostService postService;

    @Autowired
    private UserService userService;

    @PostMapping("/{postId}/like/{userId}")
    public ResponseEntity<?> likePost(@PathVariable Long postId, @PathVariable Long userId) {
        Post post = postService.getPostById(postId).orElse(null);
        User user = userService.getUser(userId);

        if (post == null || user == null) return ResponseEntity.notFound().build();
        if (postLikeService.hasUserLikedPost(user, post)) {
            return ResponseEntity.badRequest().body("Already liked");
        }

        return ResponseEntity.ok(postLikeService.likePost(user, post));
    }

    @DeleteMapping("/{postId}/unlike/{userId}")
    public ResponseEntity<?> unlikePost(@PathVariable Long postId, @PathVariable Long userId) {
        Post post = postService.getPostById(postId).orElse(null);
        User user = userService.getUser(userId);

        if (post == null || user == null) return ResponseEntity.notFound().build();
        postLikeService.unlikePost(user, post);
        return ResponseEntity.noContent().build();
    }

    @GetMapping("/{postId}/count")
    public ResponseEntity<Long> countLikes(@PathVariable Long postId) {
        return postService.getPostById(postId)
                .map(post -> ResponseEntity.ok(postLikeService.countLikes(post)))
                .orElse(ResponseEntity.notFound().build());
    }

    @GetMapping("/{postId}/likes/check")
    public ResponseEntity<Map<String, Boolean>> hasLikedPost(
            @PathVariable Long postId,
            @RequestParam Long userId  // 最好从 token/session 中获取
    ) {
        boolean liked = postLikeService.hasUserLikedPost(postId, userId);
        Map<String, Boolean> response = new HashMap<>();
        response.put("liked", liked);
        return ResponseEntity.ok(response);
    }
}