package com.github.example.pt.controller;

import cn.dev33.satoken.stp.StpUtil;
import com.github.example.pt.entity.Forum;
import com.github.example.pt.entity.Post;
import com.github.example.pt.entity.Topic;
import com.github.example.pt.entity.User;
import com.github.example.pt.service.ForumService;
import com.github.example.pt.service.PostService;
import com.github.example.pt.service.TopicService;
import com.github.example.pt.service.UserService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.Map;

import java.util.List;

@RestController
@RequestMapping("/api/topics")
public class TopicController {

    @Autowired
    private TopicService topicService;

    @Autowired
    private ForumService forumService;

    @Autowired
    private PostService postService;

    @Autowired
    private UserService UserService;

    @GetMapping
    public List<Topic> getAllTopics() {
        return topicService.getAllTopics();
    }

    @GetMapping("/forum/{forumId}")
    public ResponseEntity<List<Topic>> getTopicsByForum(@PathVariable Long forumId) {
        return forumService.getForumById(forumId)
                .map(forum -> ResponseEntity.ok(topicService.getTopicsByForum(forum)))
                .orElse(ResponseEntity.notFound().build());
    }

    @GetMapping("/{id}")
    public ResponseEntity<Topic> getTopicById(@PathVariable Long id) {
        return topicService.getTopicById(id)
                .map(ResponseEntity::ok)
                .orElse(ResponseEntity.notFound().build());
    }

    @PostMapping
    public ResponseEntity<Topic> createTopic(@RequestBody Map<String, Object> request) {
        try {
            // 获取当前登录用户
            Long userId = StpUtil.getLoginIdAsLong();
            User user = UserService.getUser(userId);
            
            // 获取请求参数
            String title = request.get("title").toString();
            String content = request.get("content").toString();
            Long forumId = Long.valueOf(request.get("forumId").toString());
            Boolean isPinned = (Boolean) request.getOrDefault("isPinned", false);
            Boolean isLocked = (Boolean) request.getOrDefault("isLocked", false);
            
            // 获取论坛
            Forum forum = forumService.getForumById(forumId)
                    .orElseThrow(() -> new RuntimeException("Forum not found"));
            
            // 创建Topic
            Topic topic = Topic.builder()
                    .title(title)
                    .forum(forum)
                    .user(user)
                    .isPinned(isPinned)
                    .isLocked(isLocked)
                    .build();
            
            Topic createdTopic = topicService.createTopic(topic);
            
            // 创建主帖内容（第一个Post）
            Post mainPost = Post.builder()
                    .content(content)
                    .topic(createdTopic)
                    .user(user)
                    .parent(null)
                    .build();
            
            postService.createPost(mainPost);
            
            return ResponseEntity.ok(createdTopic);
        } catch (Exception e) {
            e.printStackTrace();
            return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).build();
        }
    }

    @PutMapping("/{id}")
    public ResponseEntity<Topic> updateTopic(@PathVariable Long id, @RequestBody Topic topic) {
        try {
            return ResponseEntity.ok(topicService.updateTopic(id, topic));
        } catch (RuntimeException e) {
            return ResponseEntity.notFound().build();
        }
    }

    @DeleteMapping("/{id}")
    public ResponseEntity<Void> deleteTopic(@PathVariable Long id ) {
        //User user = UserService.getUser(userId);
        Topic topic = topicService.getTopicById(id).orElseThrow(() -> new RuntimeException("topic not found"));
        Long topicuserid = topic.getUser().getId();

        User user = UserService.getUser(StpUtil.getLoginIdAsLong());
        Long userId = null;
        if (user != null) {
            userId = user.getId();
            if(topicuserid.equals(userId)){
                topicService.deleteTopic(id);
                return ResponseEntity.noContent().build();
            }
        }

        return ResponseEntity.status(HttpStatus.FORBIDDEN).build();


    }

    @GetMapping("/search")
    public Page<Topic> searchTopics(
            @RequestParam String q,
            @RequestParam(required = false) Long forumId,
            @RequestParam(defaultValue = "0") int page,
            @RequestParam(defaultValue = "10") int size) {
        return topicService.searchTopics(q, forumId, page, size);
    }

}
