package com.github.example.pt.repository;

import com.github.example.pt.entity.User;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

import java.math.BigDecimal;
import java.sql.Timestamp;
import java.util.List;
import java.util.Optional;

@Repository
public interface UserRepository extends JpaRepository<User, Long> {

    // 现有的方法
    Optional<User> findByUsername(String username);
    Optional<User> findByEmail(String email);
    Optional<User> findByPasskeyIgnoreCase(String passkey);
    Optional<User> findByPersonalAccessTokenIgnoreCase(String personalAccessToken);

    // 新增的方法

    /**
     * 根据用户组ID查找用户
     */
    @Query("SELECT u FROM User u WHERE u.group.id = :groupId")
    List<User> findByGroupId(@Param("groupId") long groupId);

    /**
     * 查找分享率低于指定值的用户
     */
    @Query("SELECT u FROM User u WHERE u.karma < :ratio")
    List<User> findUsersWithLowShareRatio(@Param("ratio") BigDecimal ratio);

    /**
     * 查找分享率高于指定值的非VIP用户
     */
    @Query("SELECT u FROM User u WHERE u.karma > :ratio AND u.group.id != :vipGroupId")
    List<User> findUsersEligibleForVip(@Param("ratio") BigDecimal ratio, @Param("vipGroupId") long vipGroupId);

    /**
     * 查找最近活跃的用户（基于TransferHistory）
     * 注意：这个查询需要TransferHistory实体存在
     */
    @Query(value = "SELECT DISTINCT u.* FROM users u " +
            "JOIN transfer_histories th ON th.user_id = u.id " +
            "WHERE th.updated_at > :since", nativeQuery = true)
    List<User> findActiveUsersSince(@Param("since") Timestamp since);

    /**
     * 查找VIP用户中分享率低于阈值的用户（用于降级）
     */
    @Query("SELECT u FROM User u WHERE u.group.id = :vipGroupId AND u.karma < :ratio")
    List<User> findVipUsersWithLowRatio(@Param("vipGroupId") long vipGroupId, @Param("ratio") BigDecimal ratio);

    /**
     * 统计各用户组的用户数量
     */
    @Query("SELECT u.group.id, COUNT(u) FROM User u GROUP BY u.group.id")
    List<Object[]> countUsersByGroup();
}