package com.github.example.pt.service;

import com.github.example.pt.entity.User;
import com.github.example.pt.entity.UserGroup;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.math.RoundingMode;

@Service
@Slf4j
public class ShareRatioService {

    // 分享率阈值配置
    private static final BigDecimal DOWNLOAD_DISABLE_RATIO = new BigDecimal("0.3"); // 低于0.3禁止下载
    private static final BigDecimal VIP_UPGRADE_RATIO = new BigDecimal("2.0"); // 高于2.0自动升级VIP
    private static final long VIP_GROUP_ID = 2L; // VIP用户组ID
    private static final long NORMAL_GROUP_ID = 1L; // 普通用户组ID

    // VIP用户的上传下载系数
    private static final BigDecimal VIP_DOWNLOAD_FACTOR = new BigDecimal("0.9"); // VIP下载系数0.9
    private static final BigDecimal VIP_UPLOAD_FACTOR = new BigDecimal("1.1"); // VIP上传系数1.1

    @Autowired
    private UserService userService;

    @Autowired
    private UserGroupService userGroupService;

    /**
     * 计算用户分享率
     */
    public BigDecimal calculateShareRatio(User user) {
        if (user.getDownloaded() == 0) {
            // 如果下载量为0，返回一个很高的分享率
            return BigDecimal.valueOf(999.999);
        }
        return BigDecimal.valueOf(user.getUploaded())
                .divide(BigDecimal.valueOf(user.getDownloaded()), 3, RoundingMode.HALF_UP);
    }

    /**
     * 检查用户是否有下载权限
     */
    public boolean checkDownloadPermission(User user) {
        // 如果是VIP用户，始终允许下载
        if (isVipUser(user)) {
            return true;
        }

        // 计算分享率
        BigDecimal ratio = calculateShareRatio(user);
        user.setKarma(ratio); // 更新karma字段

        // 检查分享率是否满足下载要求
        boolean canDownload = ratio.compareTo(DOWNLOAD_DISABLE_RATIO) >= 0;

        if (!canDownload) {
            log.warn("用户 {} 分享率过低 ({})，禁止下载", user.getUsername(), ratio);
        }

        return canDownload;
    }

    /**
     * 检查并更新用户VIP状态
     */
    @Transactional
    public void checkAndUpdateVipStatus(User user) {
        BigDecimal ratio = calculateShareRatio(user);
        user.setKarma(ratio);

        // 如果已经是VIP，不需要再次升级
        if (isVipUser(user)) {
            return;
        }

        // 检查是否满足VIP升级条件
        if (ratio.compareTo(VIP_UPGRADE_RATIO) >= 0) {
            upgradeToVip(user);
            log.info("用户 {} 分享率达到 {}，自动升级为VIP", user.getUsername(), ratio);
        }
    }

    /**
     * 升级用户为VIP
     */
    @Transactional
    public void upgradeToVip(User user) {
        UserGroup vipGroup = userGroupService.getUserGroup(VIP_GROUP_ID);
        if (vipGroup != null) {
            user.setGroup(vipGroup);
            userService.save(user);
            log.info("用户 {} 已升级为VIP", user.getUsername());
        } else {
            log.error("VIP用户组不存在，无法升级用户");
        }
    }

    /**
     * 降级VIP用户（当分享率低于阈值时）
     */
    @Transactional
    public void downgradeFromVip(User user) {
        if (!isVipUser(user)) {
            return;
        }

        BigDecimal ratio = calculateShareRatio(user);
        // 如果VIP用户分享率低于1.0，降级为普通用户
        if (ratio.compareTo(BigDecimal.ONE) < 0) {
            UserGroup normalGroup = userGroupService.getUserGroup(NORMAL_GROUP_ID);
            if (normalGroup != null) {
                user.setGroup(normalGroup);
                userService.save(user);
                log.info("用户 {} 因分享率过低({})已降级为普通用户", user.getUsername(), ratio);
            }
        }
    }

    /**
     * 检查用户是否为VIP
     */
    public boolean isVipUser(User user) {
        return user.getGroup() != null && user.getGroup().getId() == VIP_GROUP_ID;
    }

    /**
     * 获取用户的下载系数
     */
    public BigDecimal getDownloadFactor(User user) {
        return isVipUser(user) ? VIP_DOWNLOAD_FACTOR : BigDecimal.ONE;
    }

    /**
     * 获取用户的上传系数
     */
    public BigDecimal getUploadFactor(User user) {
        return isVipUser(user) ? VIP_UPLOAD_FACTOR : BigDecimal.ONE;
    }

    /**
     * 应用VIP系数到实际的上传/下载量
     */
    public long applyVipDownloadFactor(User user, long downloadAmount) {
        if (isVipUser(user)) {
            return VIP_DOWNLOAD_FACTOR.multiply(BigDecimal.valueOf(downloadAmount))
                    .setScale(0, RoundingMode.HALF_UP).longValue();
        }
        return downloadAmount;
    }

    public long applyVipUploadFactor(User user, long uploadAmount) {
        if (isVipUser(user)) {
            return VIP_UPLOAD_FACTOR.multiply(BigDecimal.valueOf(uploadAmount))
                    .setScale(0, RoundingMode.HALF_UP).longValue();
        }
        return uploadAmount;
    }

    /**
     * 获取用户分享率状态描述
     */
    public String getShareRatioStatus(User user) {
        BigDecimal ratio = calculateShareRatio(user);

        if (ratio.compareTo(DOWNLOAD_DISABLE_RATIO) < 0) {
            return String.format("分享率过低 (%.3f)，无法下载", ratio);
        } else if (ratio.compareTo(VIP_UPGRADE_RATIO) >= 0) {
            return String.format("分享率优秀 (%.3f)，已达到VIP标准", ratio);
        } else {
            return String.format("分享率正常 (%.3f)", ratio);
        }
    }
}