package com.github.example.pt.service;

import com.github.example.pt.entity.Post;
import com.github.example.pt.entity.Topic;
import com.github.example.pt.repository.PostRepository;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.*;

@Service
public class PostService {

    @Autowired
    private PostRepository postRepository;

    public List<Post> getPostsByTopic(Topic topic) {
        return postRepository.findByTopicAndIsDeletedFalse(topic);
    }

    public List<Post> getPostsByUser(Long userId) {
        return postRepository.findByUserIdAndIsDeletedFalse(userId);
    }

    public Optional<Post> getPostById(Long id) {
        return postRepository.findById(id);
    }

    public Post createPost(Post post) {
        post.setCreatedAt(LocalDateTime.now());
        post.setUpdatedAt(LocalDateTime.now());
        post.setIsDeleted(false);
        return postRepository.save(post);
    }

    public Post updatePost(Long id, Post updatedPost) {
        return postRepository.findById(id).map(post -> {
            post.setContent(updatedPost.getContent());
            post.setUpdatedAt(LocalDateTime.now());
            return postRepository.save(post);
        }).orElseThrow(() -> new RuntimeException("Post not found"));
    }

    public void softDeletePost(Long id) {
        postRepository.findById(id).ifPresent(post -> {
            post.setIsDeleted(true);
            postRepository.save(post);
        });
    }

    public List<Post> getPostsByTopic(Long topicId) {
        return postRepository.findByTopicIdOrderByCreatedAtAsc(topicId);
    }

    public List<Post> buildPostTree(List<Post> flatList) {
        Map<Long, Post> map = new HashMap<>();
        List<Post> roots = new ArrayList<>();

        for (Post post : flatList) {
            map.put(post.getId(), post);
            post.setReplies(new ArrayList<>()); // 保证非 null
        }

        for (Post post : flatList) {
            if (post.getParent() != null && map.containsKey(post.getParent().getId())) {
                map.get(post.getParent().getId()).getReplies().add(post);
            } else {
                roots.add(post); // 顶层评论
            }
        }
        return roots;
    }

}
