package com.github.example.pt.controller.chat;

import com.github.example.pt.controller.chat.dto.ChatMessageDTO;
import com.github.example.pt.entity.ChatMessage;
import com.github.example.pt.entity.User;
import com.github.example.pt.repository.ChatMessageRepository;
import com.github.example.pt.service.UserService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.time.ZoneId;
import java.util.List;
import java.util.stream.Collectors;

@RestController
@RequestMapping("/chat")
@Slf4j
public class ChatHistoryController {

    private final ChatMessageRepository chatMessageRepository;
    private final UserService userService;

    @Autowired
    public ChatHistoryController(ChatMessageRepository chatMessageRepository, UserService userService) {
        this.chatMessageRepository = chatMessageRepository;
        this.userService = userService;
    }

    /**
     * GET /api/chat/history?roomId={roomId}
     * 返回指定 roomId 的所有历史消息，按时间升序排列
     *
     * @param roomId 聊天室 ID
     * @return List<ChatMessageDTO>
     */
    @GetMapping("/history")
    public List<ChatMessageDTO> getHistory(
            @RequestParam Long roomId,
            @RequestParam(defaultValue = "0")  int page,
            @RequestParam(defaultValue = "20") int size) {
        Pageable pageable = PageRequest.of(page, size, Sort.by("createdAt").ascending());

        List<ChatMessage> messages = chatMessageRepository.findByRoomIdOrderByCreatedAtAsc(roomId);

        return messages.stream().map(msg -> {
            ChatMessageDTO dto = new ChatMessageDTO();
            dto.setRoomId(msg.getRoomId());
            dto.setContent(msg.getContent());

            User user = userService.getUser(msg.getUserId());
            dto.setUsername(user!=null?user.getUsername():" unknown ");
            // 不返回头像，如果需要可以自行扩展
            dto.setAvatar(null);
            // LocalDateTime → 毫秒时间戳
            dto.setTimestamp(msg.getCreatedAt()
                    .atZone(ZoneId.systemDefault())
                    .toInstant()
                    .toEpochMilli());
            return dto;
        }).collect(Collectors.toList());
    }
}