package com.github.example.pt.service;

import com.github.example.pt.entity.User;
import com.github.example.pt.repository.UserRepository;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Optional;

@Service

public class UserService {
    @Autowired
    private UserRepository repository;

    // getUser
    @Nullable
    public User getUser(long id) {
        Optional<User> userEntity = repository.findById(id);
        return userEntity.orElse(null);
    }

    @Nullable
    public User getUserByUsername(String username) {
        Optional<User> userEntity = repository.findByUsername(username);
        return userEntity.orElse(null);
    }

    @Nullable
    public User getUserByEmail(String email) {
        Optional<User> userEntity = repository.findByEmail(email);
        return userEntity.orElse(null);
    }

    @Nullable
    public User getUserByPasskey(String passkey) {
        Optional<User> userEntity = repository.findByPasskeyIgnoreCase(passkey);
        return userEntity.orElse(null);
    }
    @Nullable
    public User getUserByPersonalAccessToken(@NotNull String personalAccessToken){
        Optional<User> userEntity = repository.findByPersonalAccessTokenIgnoreCase(personalAccessToken);
        return userEntity.orElse(null);
    }

    @NotNull
    public User save(User user) {
        return repository.save(user);
    }

    @NotNull
    public Long getUserCount() {
        return repository.count();
    }

    /**
     * 如果用户seeding_time大于86400000，生成8位数字邀请码并写入invite_slot字段
     * @param userId 用户ID
     * @return 邀请码（8位数字），否则返回null
     */
    public Integer generateInviteCodeIfEligible(long userId) {
        User user = getUser(userId);
        if (user != null && user.getSeedingTime() > 86400000L) {
            int code = (int)(Math.random() * 90000000) + 10000000; // 8位数字
            user.setInviteSlot(code);
            save(user);
            return code;
        }
        return null;
    }

    /**
     * 每2小时检查所有用户，满足条件则生成邀请码
     */
    @Scheduled(fixedRate = 2 * 60 * 60 * 1000) // 每2小时
    public void checkAndGenerateInviteCodesForAllUsers() {
        List<User> users = (List<User>) repository.findAll();
        for (User user : users) {
            if (user.getSeedingTime() > 86400000L) {
                int inviteSlot = user.getInviteSlot();
                String inviteStr = String.valueOf(inviteSlot);
                if (inviteSlot == 0) {
                    int code = (int)(Math.random() * 90000000) + 10000000;
                    user.setInviteSlot(code);
                    save(user);
                } else if (inviteStr.length() == 8) {
                    // 已有8位邀请码，跳过
                    continue;
                }
            }
        }
    }

    /**
     * 获取所有用户
     */
    public Iterable<User> getAllUsers() {
        return repository.findAll();
    }
}
