package com.github.example.pt.service;

import com.github.example.pt.entity.Peer;
import com.github.example.pt.entity.Torrent;
import com.github.example.pt.entity.TransferHistory;
import com.github.example.pt.entity.User;
import com.github.example.pt.exception.AnnounceBusyException;
import com.github.example.pt.type.AnnounceEventType;
import com.github.example.pt.util.ExecutorUtil;
import com.github.example.pt.util.HibernateSessionUtil;
import jakarta.persistence.EntityManagerFactory;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.sql.Timestamp;
import java.time.Instant;
import java.util.NoSuchElementException;
import java.util.concurrent.BlockingDeque;
import java.util.concurrent.LinkedBlockingDeque;

@Service
@Slf4j
public class AnnounceService {
    private final BlockingDeque<AnnounceTask> taskQueue = new LinkedBlockingDeque<>(40960);
    @Autowired
    private ExecutorUtil executor;
    @Autowired
    private UserService userService;
    @Autowired
    private PeerService peerService;
    @Autowired
    private TorrentService torrentService;
    @Autowired
    private EntityManagerFactory entityManagerFactory;
    @Autowired
    private AnnouncePerformanceMonitorService monitorService;
    @Autowired
    private TransferHistoryService transferHistoryService;
    @Autowired
    private HibernateSessionUtil sessionUtil;
    @Autowired
    private ShareRatioService shareRatioService;


    public AnnounceService() {
        Thread announceHandleThread = new Thread(() -> {
            while (true) {
                try {
                    AnnounceTask announceTask = taskQueue.take();
                    boolean participate = sessionUtil.bindToThread();
                    try {
                        long start = System.nanoTime();
                        handleTask(announceTask);
                        monitorService.recordJobStats(System.nanoTime() - start);
                    } catch (Exception e) {
                        log.error("Error handling task: {}", announceTask, e);
                    } finally {
                        sessionUtil.closeFromThread(participate);
                    }
                } catch (InterruptedException e) {
                    log.error("Announce handling thread interrupted", e);
                }
            }
        });
        announceHandleThread.setName("Announce Handling");
        announceHandleThread.setDaemon(true);
        announceHandleThread.start();
    }

    public void schedule(@NotNull AnnounceTask announceTask) throws AnnounceBusyException {
        if (!this.taskQueue.offer(announceTask))
            throw new AnnounceBusyException();
    }

    @Transactional
    void handleTask(AnnounceTask task) throws NoSuchElementException {
        // Multi-threaded
        User user = userService.getUser(task.userId());
        if (user == null) throw new IllegalStateException("User not exists anymore");
        Torrent torrent = torrentService.getTorrent(task.torrentId());
        if (torrent == null) throw new IllegalStateException("Torrent not exists anymore");

        // Register torrent into peers
        Peer peer = peerService.getPeer(task.ip(), task.port(), task.infoHash());
        if (peer == null) {
            peer = createNewPeer(task, user);
        }

        long lastUploaded = peer.getUploaded();
        long lastDownload = peer.getDownloaded();
        long uploadedOffset = task.uploaded() - lastUploaded;
        long downloadedOffset = task.downloaded() - lastDownload;
        Timestamp lastUpdateAt = peer.getUpdateAt(); // 修正：应该是peer的更新时间

        if (uploadedOffset < 0) uploadedOffset = task.uploaded();
        if (downloadedOffset < 0) downloadedOffset = task.downloaded();

        long announceInterval = Instant.now().toEpochMilli() - lastUpdateAt.toInstant().toEpochMilli();
        peer.setUploaded(task.uploaded());
        peer.setDownloaded(task.downloaded());
        peer.setLeft(task.left());
        peer.setSeeder(task.left() == 0);
        peer.setUpdateAt(Timestamp.from(Instant.now()));
        peer.setSeedingTime((peer.getSeedingTime() + announceInterval) / 86400000); // 转换为天
        peer.setPartialSeeder(task.event() == AnnounceEventType.PAUSED);

        // Update user peer speed
        long bytesPerSecondUploading = announceInterval > 0 ? uploadedOffset / (announceInterval / 1000) : 0;
        long bytesPerSecondDownloading = announceInterval > 0 ? downloadedOffset / (announceInterval / 1000) : 0;
        peer.setUploadSpeed(bytesPerSecondUploading);
        peer.setDownloadSpeed(bytesPerSecondDownloading);
        peer = peerService.save(peer);

        // Update real user data
        user.setRealDownloaded(user.getRealDownloaded() + downloadedOffset);
        user.setRealUploaded(user.getRealUploaded() + uploadedOffset);

        // 计算用户的分享率
        BigDecimal karma = shareRatioService.calculateShareRatio(user);
        user.setKarma(karma);

        // 检查并更新VIP状态
        shareRatioService.checkAndUpdateVipStatus(user);

        // Apply user group promotion policy
        long promotionUploadOffset = (long) user.getGroup().getPromotionPolicy().applyUploadRatio(uploadedOffset);
        long promotionDownloadOffset = (long) user.getGroup().getPromotionPolicy().applyDownloadRatio(downloadedOffset);

        // Apply torrent promotion policy
        promotionUploadOffset = (long) torrent.getPromotionPolicy().applyUploadRatio(promotionUploadOffset);
        promotionDownloadOffset = (long) torrent.getPromotionPolicy().applyDownloadRatio(promotionDownloadOffset);

        // 应用VIP系数（在所有其他promotion之后）
        if (shareRatioService.isVipUser(user)) {
            // VIP用户：下载量乘以0.9，上传量乘以1.1
            promotionDownloadOffset = shareRatioService.applyVipDownloadFactor(user, promotionDownloadOffset);
            promotionUploadOffset = shareRatioService.applyVipUploadFactor(user, promotionUploadOffset);
            log.debug("用户 {} 为VIP，应用VIP系数后：上传增量={}，下载增量={}",
                    user.getUsername(), promotionUploadOffset, promotionDownloadOffset);
        }

        user.setUploaded(user.getUploaded() + promotionUploadOffset);
        user.setDownloaded(user.getDownloaded() + promotionDownloadOffset);
        user.setSeedingTime(user.getSeedingTime() + announceInterval);
        user = userService.save(user);

        // 更新TransferHistory
        TransferHistory transferHistory = transferHistoryService.getTransferHistory(user, torrent);
        if (transferHistory != null) {
            long torrentLeft = transferHistory.getLeft();
            if (torrentLeft != 0 && task.left() == 0) {
                transferHistory.setHaveCompleteHistory(true);
            }
            transferHistory.setUpdatedAt(Timestamp.from(Instant.now()));
            transferHistory.setLeft(task.left());
            transferHistory.setUploaded(transferHistory.getUploaded() + promotionUploadOffset);
            transferHistory.setDownloaded(transferHistory.getDownloaded() + promotionDownloadOffset);
            transferHistory.setActualUploaded(transferHistory.getActualUploaded() + uploadedOffset);
            transferHistory.setActualDownloaded(transferHistory.getActualDownloaded() + downloadedOffset);
            transferHistory.setUploadSpeed(bytesPerSecondUploading);
            transferHistory.setDownloadSpeed(bytesPerSecondDownloading);
        } else {
            transferHistory = new TransferHistory(0, user, torrent,
                    task.left(), Timestamp.from(Instant.now()),
                    Timestamp.from(Instant.now()),
                    promotionUploadOffset, promotionDownloadOffset, uploadedOffset, downloadedOffset,
                    bytesPerSecondUploading, bytesPerSecondDownloading, task.event(), false);
        }
        transferHistoryService.save(transferHistory);
        torrentService.save(torrent);

        if (task.event() == AnnounceEventType.STOPPED) {
            if (peer.getId() != 0) {
                peerService.delete(peer);
            }
        }
    }


    @NotNull
    private Peer createNewPeer(AnnounceTask task, User user) {
        return new Peer(
                0,
                task.ip(),
                task.port(),
                task.infoHash(),
                task.peerId(),
                task.userAgent(),
                task.uploaded(),
                task.downloaded(),
                task.left(),
                task.left() == 0,
                task.event() == AnnounceEventType.PAUSED,
                task.passKey(),
                Timestamp.from(Instant.now()),
                0, 0,
                0,
                user
        );
    }

    public record AnnounceTask(
            @NotNull String ip, int port, @NotNull String infoHash, @NotNull String peerId,
            long uploaded, long downloaded, long left, @NotNull AnnounceEventType event,
            int numWant, long userId, boolean compact, boolean noPeerId,
            boolean supportCrypto, int redundant, String userAgent, String passKey, long torrentId
    ) {

    }
}
