package com.ruoyi.web.controller.post.service.impl;

import java.util.List;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import com.ruoyi.web.controller.post.mapper.PostContentMapper;
import com.ruoyi.web.controller.post.mapper.PostPaymentMapper;
import com.ruoyi.web.controller.post.mapper.PostPromotionPlanMapper;
import com.ruoyi.web.controller.post.domain.Post;
import com.ruoyi.web.controller.post.domain.PostPayment;
import com.ruoyi.web.controller.post.domain.PostPromotionPlan;
import com.ruoyi.web.controller.post.service.IPostService;

import java.util.Date;
import java.util.Calendar;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import java.util.ArrayList;

/**
 * 帖子Service业务层处理
 * 
 * @author thunderhub
 */
@Service
public class PostServiceImpl implements IPostService
{
    @Autowired
    private PostContentMapper postMapper;

    @Autowired
    private PostPaymentMapper postPaymentMapper;

    @Autowired
    private PostPromotionPlanMapper postPromotionPlanMapper;

    /**
     * 查询帖子信息
     * 
     * @param postId 帖子ID
     * @return 帖子信息
     */
    @Override
    public Post selectPostById(Long postId)
    {
        return postMapper.selectPostById(postId);
    }

    /**
     * 查询帖子列表
     * 
     * @param post 帖子信息
     * @return 帖子集合
     */
    @Override
    public List<Post> selectPostList(Post post)
    {
        return postMapper.selectPostList(post);
    }

    /**
     * 查询作者的其他帖子
     * 
     * @param authorId 作者ID
     * @param postId 当前帖子ID（排除）
     * @param limit 查询数量
     * @return 帖子集合
     */
    @Override
    public List<Post> selectAuthorOtherPosts(Long authorId, Long postId, int limit)
    {
        return postMapper.selectAuthorOtherPosts(authorId, postId, limit);
    }

    /**
     * 查询相似标签的帖子
     * 
     * @param tags 标签列表
     * @param postId 当前帖子ID（排除）
     * @param limit 查询数量
     * @return 帖子集合
     */
    @Override
    public List<Post> selectSimilarTagsPosts(String tags, Long postId, int limit)
    {
        if (tags == null || tags.trim().isEmpty()) {
            return List.of();
        }
        
        return postMapper.selectSimilarTagsPosts(tags, postId, limit);
    }

    /**
     * 新增帖子
     * 
     * @param post 帖子信息
     * @return 结果
     */
    @Override
    public int insertPost(Post post)
    {
        return postMapper.insertPost(post);
    }

    /**
     * 修改帖子
     * 
     * @param post 帖子信息
     * @return 结果
     */
    @Override
    public int updatePost(Post post)
    {
        return postMapper.updatePost(post);
    }

    /**
     * 删除帖子信息
     * 
     * @param postId 帖子ID
     * @return 结果
     */
    @Override
    public int deletePostById(Long postId)
    {
        return postMapper.deletePostById(postId);
    }

    /**
     * 批量删除帖子信息
     * 
     * @param postIds 需要删除的帖子ID
     * @return 结果
     */
    @Override
    public int deletePostByIds(Long[] postIds)
    {
        return postMapper.deletePostByIds(postIds);
    }

    /**
     * 更新帖子浏览量
     * 
     * @param postId 帖子ID
     * @return 结果
     */
    @Override
    public int updatePostViews(Long postId)
    {
        return postMapper.updatePostViews(postId);
    }

    /**
     * 更新帖子评论数
     * 
     * @param postId 帖子ID
     * @param count 增加/减少的数量
     * @return 结果
     */
    @Override
    public int updatePostComments(Long postId, int count)
    {
        Post post = new Post();
        post.setPostId(postId);
        post.setComments((long) count);
        return postMapper.updatePostComments(post);
    }

    /**
     * 更新帖子收藏数
     * 
     * @param postId 帖子ID
     * @param count 增加/减少的数量
     * @return 结果
     */
    @Override
    public int updatePostFavorites(Long postId, int count)
    {
        Post post = new Post();
        post.setPostId(postId);
        post.setFavorites((long) count);
        return postMapper.updatePostFavorites(post);
    }

    /**
     * 更新帖子点赞数
     * 
     * @param postId 帖子ID
     * @param count 增加/减少的数量
     * @return 结果
     */
    @Override
    public int updatePostLikes(Long postId, int count)
    {
        Post post = new Post();
        post.setPostId(postId);
        post.setLikes((long) count);
        return postMapper.updatePostLikes(post);
    }

    /**
     * 检查并处理过期的推广帖子
     * 
     * @return 处理的帖子数量
     */
    @Override
    public int checkAndHandleExpiredPromotions()
    {
        Logger logger = LoggerFactory.getLogger(PostServiceImpl.class);
        int processedCount = 0;
        
        try {
            // 查询所有过期的推广支付记录
            List<PostPayment> expiredPayments = postPaymentMapper.selectExpiredPromotionPayments();
            
            for (PostPayment payment : expiredPayments) {
                // 获取推广计划信息
                PostPromotionPlan plan = postPromotionPlanMapper.selectPostPromotionPlanById(payment.getPlanId());
                if (plan == null) {
                    continue;
                }
                
                // 计算推广到期时间
                Calendar cal = Calendar.getInstance();
                cal.setTime(payment.getPaymentTime());
                cal.add(Calendar.DAY_OF_MONTH, plan.getDuration());
                Date expireTime = cal.getTime();
                
                // 检查是否已过期
                if (new Date().after(expireTime)) {
                    // 取消帖子推广
                    int result = cancelPostPromotion(payment.getPostId());
                    if (result > 0) {
                        processedCount++;
                        logger.info("帖子ID: {} 的推广已过期，已自动取消", payment.getPostId());
                    }
                }
            }
        } catch (Exception e) {
            logger.error("检查推广时效性时发生错误", e);
        }
        
        return processedCount;
    }

    /**
     * 取消帖子推广
     * 
     * @param postId 帖子ID
     * @return 结果
     */
    @Override
    public int cancelPostPromotion(Long postId)
    {
        Post post = new Post();
        post.setPostId(postId);
        post.setPromotionPlanId(null);
        return postMapper.updatePost(post);
    }

    /**
     * 获取帖子推荐列表（包含作者其他帖子和相似标签帖子）
     * 
     * @param authorId 作者ID
     * @param tags 标签列表
     * @param postId 当前帖子ID（排除）
     * @param maxCount 最大推荐数量
     * @return 推荐帖子集合
     */
    @Override
    public List<Post> getRecommendedPosts(Long authorId, String tags, Long postId, int maxCount)
    {
        List<Post> recommendedPosts = new ArrayList<>();
        
        // 1. 获取作者的其他帖子（最多5个）
        List<Post> authorPosts = selectAuthorOtherPosts(authorId, postId, 5);
        recommendedPosts.addAll(authorPosts);
        
        // 2. 如果作者帖子不足maxCount个，则补充相似标签的帖子
        if (recommendedPosts.size() < maxCount && tags != null && !tags.trim().isEmpty()) {
            int remainingCount = maxCount - recommendedPosts.size();
            List<Post> similarPosts = selectSimilarTagsPosts(tags, postId, remainingCount + 5); // 多查询一些以便去重
            
            // 去重：排除已经在推荐列表中的帖子
            for (Post similarPost : similarPosts) {
                if (recommendedPosts.size() >= maxCount) {
                    break;
                }
                
                boolean isDuplicate = false;
                for (Post existingPost : recommendedPosts) {
                    if (existingPost.getPostId().equals(similarPost.getPostId())) {
                        isDuplicate = true;
                        break;
                    }
                }
                
                if (!isDuplicate) {
                    recommendedPosts.add(similarPost);
                }
            }
        }
        
        // 3. 限制最大数量
        if (recommendedPosts.size() > maxCount) {
            recommendedPosts = recommendedPosts.subList(0, maxCount);
        }
        
        return recommendedPosts;
    }
} 