/*
 * Decompiled with CFR 0.152.
 */
package com.turn.ttorrent.common.protocol;

import com.turn.ttorrent.Constants;
import com.turn.ttorrent.common.TorrentInfo;
import java.nio.ByteBuffer;
import java.text.ParseException;
import java.util.BitSet;

public abstract class PeerMessage {
    public static final int MESSAGE_LENGTH_FIELD_SIZE = 4;
    private final Type type;
    private final ByteBuffer data;

    private PeerMessage(Type type, ByteBuffer data) {
        this.type = type;
        this.data = data;
        this.data.rewind();
    }

    public Type getType() {
        return this.type;
    }

    public ByteBuffer getData() {
        return this.data.duplicate();
    }

    public PeerMessage validate(TorrentInfo torrent) throws MessageValidationException {
        return this;
    }

    public String toString() {
        return this.getType().name();
    }

    public static PeerMessage parse(ByteBuffer buffer, TorrentInfo torrent) throws ParseException {
        int length = buffer.getInt();
        if (length == 0) {
            return KeepAliveMessage.parse(buffer, torrent);
        }
        if (length != buffer.remaining()) {
            throw new ParseException("Message size did not match announced size!", 0);
        }
        Type type = Type.get(buffer.get());
        if (type == null) {
            throw new ParseException("Unknown message ID!", buffer.position() - 1);
        }
        switch (type) {
            case CHOKE: {
                return ChokeMessage.parse(buffer.slice(), torrent);
            }
            case UNCHOKE: {
                return UnchokeMessage.parse(buffer.slice(), torrent);
            }
            case INTERESTED: {
                return InterestedMessage.parse(buffer.slice(), torrent);
            }
            case NOT_INTERESTED: {
                return NotInterestedMessage.parse(buffer.slice(), torrent);
            }
            case HAVE: {
                return HaveMessage.parse(buffer.slice(), torrent);
            }
            case BITFIELD: {
                return BitfieldMessage.parse(buffer.slice(), torrent);
            }
            case REQUEST: {
                return RequestMessage.parse(buffer.slice(), torrent);
            }
            case PIECE: {
                return PieceMessage.parse(buffer.slice(), torrent);
            }
            case CANCEL: {
                return CancelMessage.parse(buffer.slice(), torrent);
            }
        }
        throw new IllegalStateException("Message type should have been properly defined by now.");
    }

    public static class CancelMessage
    extends PeerMessage {
        private static final int BASE_SIZE = 13;
        private int piece;
        private int offset;
        private int length;

        private CancelMessage(ByteBuffer buffer, int piece, int offset, int length) {
            super(Type.CANCEL, buffer);
            this.piece = piece;
            this.offset = offset;
            this.length = length;
        }

        public int getPiece() {
            return this.piece;
        }

        public int getOffset() {
            return this.offset;
        }

        public int getLength() {
            return this.length;
        }

        @Override
        public CancelMessage validate(TorrentInfo torrent) throws MessageValidationException {
            if (this.piece >= 0 && this.piece < torrent.getPieceCount() && (long)(this.offset + this.length) <= torrent.getPieceSize(this.piece)) {
                return this;
            }
            throw new MessageValidationException(this);
        }

        public static CancelMessage parse(ByteBuffer buffer, TorrentInfo torrent) throws MessageValidationException {
            int piece = buffer.getInt();
            int offset = buffer.getInt();
            int length = buffer.getInt();
            return new CancelMessage(buffer, piece, offset, length).validate(torrent);
        }

        public static CancelMessage craft(int piece, int offset, int length) {
            ByteBuffer buffer = ByteBuffer.allocate(17);
            buffer.putInt(13);
            buffer.put(Type.CANCEL.getTypeByte());
            buffer.putInt(piece);
            buffer.putInt(offset);
            buffer.putInt(length);
            return new CancelMessage(buffer, piece, offset, length);
        }

        @Override
        public String toString() {
            return super.toString() + " #" + this.getPiece() + " (" + this.getLength() + "@" + this.getOffset() + ")";
        }
    }

    public static class PieceMessage
    extends PeerMessage {
        private static final int BASE_SIZE = 9;
        private int piece;
        private int offset;
        private ByteBuffer block;

        private PieceMessage(ByteBuffer buffer, int piece, int offset, ByteBuffer block) {
            super(Type.PIECE, buffer);
            this.piece = piece;
            this.offset = offset;
            this.block = block;
        }

        public int getPiece() {
            return this.piece;
        }

        public int getOffset() {
            return this.offset;
        }

        public ByteBuffer getBlock() {
            return this.block;
        }

        @Override
        public PieceMessage validate(TorrentInfo torrent) throws MessageValidationException {
            if (this.piece >= 0 && this.piece < torrent.getPieceCount() && (long)(this.offset + this.block.limit()) <= torrent.getPieceSize(this.piece)) {
                return this;
            }
            throw new MessageValidationException(this);
        }

        public static PieceMessage parse(ByteBuffer buffer, TorrentInfo torrent) throws MessageValidationException {
            int piece = buffer.getInt();
            int offset = buffer.getInt();
            ByteBuffer block = buffer.slice();
            return new PieceMessage(buffer, piece, offset, block).validate(torrent);
        }

        public static PieceMessage craft(int piece, int offset, ByteBuffer buffer) {
            return new PieceMessage(buffer, piece, offset, Constants.EMPTY_BUFFER);
        }

        public static ByteBuffer createBufferWithHeaderForMessage(int piece, int offset, int blockSize) {
            ByteBuffer result = ByteBuffer.allocate(13 + blockSize);
            result.putInt(9 + blockSize);
            result.put(Type.PIECE.getTypeByte());
            result.putInt(piece);
            result.putInt(offset);
            return result;
        }

        @Override
        public String toString() {
            return super.toString() + " #" + this.getPiece() + " (" + this.getBlock().capacity() + "@" + this.getOffset() + ")";
        }
    }

    public static class RequestMessage
    extends PeerMessage {
        private static final int BASE_SIZE = 13;
        public static final int DEFAULT_REQUEST_SIZE = 16384;
        public static final int MAX_REQUEST_SIZE = 131072;
        private int piece;
        private int offset;
        private int length;
        private long mySendTime;

        private RequestMessage(ByteBuffer buffer, int piece, int offset, int length) {
            super(Type.REQUEST, buffer);
            this.piece = piece;
            this.offset = offset;
            this.length = length;
            this.mySendTime = System.currentTimeMillis();
        }

        public int getPiece() {
            return this.piece;
        }

        public int getOffset() {
            return this.offset;
        }

        public int getLength() {
            return this.length;
        }

        public long getSendTime() {
            return this.mySendTime;
        }

        public void renew() {
            this.mySendTime = System.currentTimeMillis();
        }

        @Override
        public RequestMessage validate(TorrentInfo torrent) throws MessageValidationException {
            if (this.piece >= 0 && this.piece < torrent.getPieceCount() && (long)(this.offset + this.length) <= torrent.getPieceSize(this.piece)) {
                return this;
            }
            throw new MessageValidationException(this);
        }

        public static RequestMessage parse(ByteBuffer buffer, TorrentInfo torrent) throws MessageValidationException {
            int piece = buffer.getInt();
            int offset = buffer.getInt();
            int length = buffer.getInt();
            return new RequestMessage(buffer, piece, offset, length).validate(torrent);
        }

        public static RequestMessage craft(int piece, int offset, int length) {
            ByteBuffer buffer = ByteBuffer.allocate(17);
            buffer.putInt(13);
            buffer.put(Type.REQUEST.getTypeByte());
            buffer.putInt(piece);
            buffer.putInt(offset);
            buffer.putInt(length);
            return new RequestMessage(buffer, piece, offset, length);
        }

        @Override
        public String toString() {
            return super.toString() + " #" + this.getPiece() + " (" + this.getLength() + "@" + this.getOffset() + ")";
        }
    }

    public static class BitfieldMessage
    extends PeerMessage {
        private static final int BASE_SIZE = 1;
        private BitSet bitfield;

        private BitfieldMessage(ByteBuffer buffer, BitSet bitfield) {
            super(Type.BITFIELD, buffer);
            this.bitfield = bitfield;
        }

        public BitSet getBitfield() {
            return this.bitfield;
        }

        @Override
        public BitfieldMessage validate(TorrentInfo torrent) throws MessageValidationException {
            if (this.bitfield.length() <= torrent.getPieceCount()) {
                return this;
            }
            throw new MessageValidationException(this);
        }

        public static BitfieldMessage parse(ByteBuffer buffer, TorrentInfo torrent) throws MessageValidationException {
            BitSet bitfield = new BitSet(buffer.remaining() * 8);
            for (int i = 0; i < buffer.remaining() * 8; ++i) {
                if ((buffer.get(i / 8) & 1 << 7 - i % 8) <= 0) continue;
                bitfield.set(i);
            }
            return new BitfieldMessage(buffer, bitfield).validate(torrent);
        }

        public static BitfieldMessage craft(BitSet availablePieces) {
            int len = availablePieces.length() / 8;
            if (availablePieces.length() % 8 > 0) {
                ++len;
            }
            byte[] bitfield = new byte[len];
            int i = availablePieces.nextSetBit(0);
            while (i >= 0) {
                int n = i / 8;
                bitfield[n] = (byte)(bitfield[n] | 1 << 7 - i % 8);
                i = availablePieces.nextSetBit(i + 1);
            }
            ByteBuffer buffer = ByteBuffer.allocate(5 + bitfield.length);
            buffer.putInt(1 + bitfield.length);
            buffer.put(Type.BITFIELD.getTypeByte());
            buffer.put(ByteBuffer.wrap(bitfield));
            return new BitfieldMessage(buffer, availablePieces);
        }

        @Override
        public String toString() {
            return super.toString() + " " + this.getBitfield().cardinality();
        }
    }

    public static class HaveMessage
    extends PeerMessage {
        private static final int BASE_SIZE = 5;
        private int piece;

        private HaveMessage(ByteBuffer buffer, int piece) {
            super(Type.HAVE, buffer);
            this.piece = piece;
        }

        public int getPieceIndex() {
            return this.piece;
        }

        @Override
        public HaveMessage validate(TorrentInfo torrent) throws MessageValidationException {
            if (this.piece >= 0 && this.piece < torrent.getPieceCount()) {
                return this;
            }
            throw new MessageValidationException(this);
        }

        public static HaveMessage parse(ByteBuffer buffer, TorrentInfo torrent) throws MessageValidationException {
            return new HaveMessage(buffer, buffer.getInt()).validate(torrent);
        }

        public static HaveMessage craft(int piece) {
            ByteBuffer buffer = ByteBuffer.allocate(9);
            buffer.putInt(5);
            buffer.put(Type.HAVE.getTypeByte());
            buffer.putInt(piece);
            return new HaveMessage(buffer, piece);
        }

        @Override
        public String toString() {
            return super.toString() + " #" + this.getPieceIndex();
        }
    }

    public static class NotInterestedMessage
    extends PeerMessage {
        private static final int BASE_SIZE = 1;

        private NotInterestedMessage(ByteBuffer buffer) {
            super(Type.NOT_INTERESTED, buffer);
        }

        public static NotInterestedMessage parse(ByteBuffer buffer, TorrentInfo torrent) throws MessageValidationException {
            return (NotInterestedMessage)new NotInterestedMessage(buffer).validate(torrent);
        }

        public static NotInterestedMessage craft() {
            ByteBuffer buffer = ByteBuffer.allocate(5);
            buffer.putInt(1);
            buffer.put(Type.NOT_INTERESTED.getTypeByte());
            return new NotInterestedMessage(buffer);
        }
    }

    public static class InterestedMessage
    extends PeerMessage {
        private static final int BASE_SIZE = 1;

        private InterestedMessage(ByteBuffer buffer) {
            super(Type.INTERESTED, buffer);
        }

        public static InterestedMessage parse(ByteBuffer buffer, TorrentInfo torrent) throws MessageValidationException {
            return (InterestedMessage)new InterestedMessage(buffer).validate(torrent);
        }

        public static InterestedMessage craft() {
            ByteBuffer buffer = ByteBuffer.allocate(5);
            buffer.putInt(1);
            buffer.put(Type.INTERESTED.getTypeByte());
            return new InterestedMessage(buffer);
        }
    }

    public static class UnchokeMessage
    extends PeerMessage {
        private static final int BASE_SIZE = 1;

        private UnchokeMessage(ByteBuffer buffer) {
            super(Type.UNCHOKE, buffer);
        }

        public static UnchokeMessage parse(ByteBuffer buffer, TorrentInfo torrent) throws MessageValidationException {
            return (UnchokeMessage)new UnchokeMessage(buffer).validate(torrent);
        }

        public static UnchokeMessage craft() {
            ByteBuffer buffer = ByteBuffer.allocate(5);
            buffer.putInt(1);
            buffer.put(Type.UNCHOKE.getTypeByte());
            return new UnchokeMessage(buffer);
        }
    }

    public static class ChokeMessage
    extends PeerMessage {
        private static final int BASE_SIZE = 1;

        private ChokeMessage(ByteBuffer buffer) {
            super(Type.CHOKE, buffer);
        }

        public static ChokeMessage parse(ByteBuffer buffer, TorrentInfo torrent) throws MessageValidationException {
            return (ChokeMessage)new ChokeMessage(buffer).validate(torrent);
        }

        public static ChokeMessage craft() {
            ByteBuffer buffer = ByteBuffer.allocate(5);
            buffer.putInt(1);
            buffer.put(Type.CHOKE.getTypeByte());
            return new ChokeMessage(buffer);
        }
    }

    public static class KeepAliveMessage
    extends PeerMessage {
        private static final int BASE_SIZE = 0;

        private KeepAliveMessage(ByteBuffer buffer) {
            super(Type.KEEP_ALIVE, buffer);
        }

        public static KeepAliveMessage parse(ByteBuffer buffer, TorrentInfo torrent) throws MessageValidationException {
            return (KeepAliveMessage)new KeepAliveMessage(buffer).validate(torrent);
        }

        public static KeepAliveMessage craft() {
            ByteBuffer buffer = ByteBuffer.allocate(4);
            buffer.putInt(0);
            return new KeepAliveMessage(buffer);
        }
    }

    public static class MessageValidationException
    extends ParseException {
        static final long serialVersionUID = -1L;

        public MessageValidationException(PeerMessage m) {
            super("Message " + m + " is not valid!", 0);
        }
    }

    public static enum Type {
        KEEP_ALIVE(-1),
        CHOKE(0),
        UNCHOKE(1),
        INTERESTED(2),
        NOT_INTERESTED(3),
        HAVE(4),
        BITFIELD(5),
        REQUEST(6),
        PIECE(7),
        CANCEL(8);

        private byte id;

        private Type(int id) {
            this.id = (byte)id;
        }

        public boolean equals(byte c) {
            return this.id == c;
        }

        public byte getTypeByte() {
            return this.id;
        }

        public static Type get(byte c) {
            for (Type t : Type.values()) {
                if (!t.equals(c)) continue;
                return t;
            }
            return null;
        }
    }
}

