package com.ruoyi.web.controller.post.controller;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.math.BigDecimal;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import com.github.pagehelper.PageHelper;
import com.ruoyi.common.annotation.Anonymous;
import com.ruoyi.common.core.controller.BaseController;
import com.ruoyi.common.core.domain.AjaxResult;
import com.ruoyi.common.core.page.TableDataInfo;
import com.ruoyi.web.controller.post.domain.Post;
import com.ruoyi.web.controller.post.domain.PostComment;
import com.ruoyi.web.controller.post.domain.PostFavorite;
import com.ruoyi.web.controller.post.domain.PostTag;
import com.ruoyi.web.controller.post.domain.PostPayment;
import com.ruoyi.web.controller.post.domain.PostReport;
import com.ruoyi.web.controller.post.service.IPostCommentService;
import com.ruoyi.web.controller.post.service.IPostFavoriteService;
import com.ruoyi.web.controller.post.service.IPostService;
import com.ruoyi.web.controller.post.service.IPostTagService;
import com.ruoyi.web.controller.post.mapper.PostPaymentMapper;
import com.ruoyi.web.controller.post.mapper.PostReportMapper;

/**
 * 帖子中心控制器，面向前端用户
 * 
 * @author thunderhub
 */
@RestController
@RequestMapping("/post-center")
public class PostCenterController extends BaseController
{
    @Autowired
    private IPostService postService;
    
    @Autowired
    private IPostTagService postTagService;
    
    @Autowired
    private IPostCommentService postCommentService;
    
    @Autowired
    private IPostFavoriteService postFavoriteService;
    
    @Autowired
    private PostPaymentMapper postPaymentMapper;
    
    @Autowired
    private PostReportMapper postReportMapper;

    /**
     * 获取帖子列表
     */
    @Anonymous
    @GetMapping("/list")
    public TableDataInfo list(Post post, 
                             @RequestParam(value = "pageNum", defaultValue = "1") Integer pageNum,
                             @RequestParam(value = "pageSize", defaultValue = "10") Integer pageSize,
                             @RequestParam(value = "tags", required = false) String tags) 
    {
        // 手动设置分页参数 - 确保分页参数正确传递
        PageHelper.startPage(pageNum, pageSize);
        
        // 只查询状态为正常的帖子
        if (post.getStatus() == null) {
            post.setStatus("1");
        }
        
        // 如果有标签筛选参数，设置到查询条件中
        if (tags != null && !tags.trim().isEmpty() && !"all".equals(tags)) {
            post.setTags(tags);
        }
        
        List<Post> list = postService.selectPostList(post);
        return getDataTable(list);
    }

    /**
     * 获取推广帖子列表（用于轮播展示）
     */
    @Anonymous
    @GetMapping("/promotion")
    public AjaxResult getPromotionPosts()
    {
        Post query = new Post();
        query.setStatus("1"); // 只查询已发布的帖子
        
        List<Post> allPosts = postService.selectPostList(query);
        
        // 筛选出购买了最高等级推广（promotionPlanId = 1）的帖子用于轮播
        List<Post> carouselPosts = allPosts.stream()
                .filter(post -> post.getPromotionPlanId() != null && post.getPromotionPlanId().equals(1L))
                .limit(5) // 限制轮播数量
                .collect(java.util.stream.Collectors.toList());
        
        return success(carouselPosts);
    }

    /**
     * 获取帖子详情
     */
    @Anonymous
    @GetMapping("/{postId}")
    public AjaxResult getDetail(@PathVariable Long postId)
    {
        Map<String, Object> result = new HashMap<>();
        
        // 更新浏览量
        postService.updatePostViews(postId);
        
        // 获取帖子信息
        Post post = postService.selectPostById(postId);
        if (post == null || !"1".equals(post.getStatus())) {
            return error("帖子不存在或已下架");
        }
        result.put("post", post);
        
        // 获取帖子标签
        List<PostTag> tags = postTagService.selectPostTagsByPostId(postId);
        result.put("tags", tags);
        
        // 获取评论，只返回顶级评论，限制10条
        List<PostComment> comments = postCommentService.selectCommentsByPostId(postId, 10);
        
        // 获取每个评论的回复
        List<Map<String, Object>> commentList = new ArrayList<>();
        for (PostComment comment : comments) {
            if ("1".equals(comment.getStatus())) { // 只返回正常状态的评论
                Map<String, Object> commentMap = new HashMap<>();
                commentMap.put("comment", comment);
                
                List<PostComment> replies = postCommentService.selectCommentsByParentId(comment.getCommentId());
                // 过滤回复，只返回正常状态的回复
                List<PostComment> validReplies = new ArrayList<>();
                for (PostComment reply : replies) {
                    if ("1".equals(reply.getStatus())) {
                        validReplies.add(reply);
                    }
                }
                commentMap.put("replies", validReplies);
                commentList.add(commentMap);
            }
        }
        result.put("comments", commentList);
        
        // 获取推荐帖子（最多9个）
        List<Post> recommendedPosts = postService.getRecommendedPosts(post.getAuthorId(), post.getTags(), postId, 9);
        result.put("recommendedPosts", recommendedPosts);
        
        // 判断当前用户是否已收藏
        Long userId = null;
        try {
            userId = getUserId();
        } catch (Exception e) {
            // 用户未登录，忽略异常
        }
        
        if (userId != null) {
            PostFavorite favorite = postFavoriteService.selectPostFavoriteByPostIdAndUserId(postId, userId);
            result.put("favorited", favorite != null && "0".equals(favorite.getStatus()));
        } else {
            result.put("favorited", false);
        }
        
        return success(result);
    }

    /**
     * 添加评论
     */
    @PostMapping("/comment")
    public AjaxResult addComment(@RequestBody PostComment postComment)
    {
        if (postComment.getPostId() == null) {
            return error("帖子ID不能为空");
        }
        
        if (postComment.getContent() == null || postComment.getContent().trim().isEmpty()) {
            return error("评论内容不能为空");
        }
        
        // 检查用户是否登录
        Long userId = null;
        String username = null;
        try {
            userId = getUserId();
            username = getUsername();
        } catch (Exception e) {
            return error("请先登录后再评论");
        }
        
        if (userId == null) {
            return error("请先登录后再评论");
        }
        
        postComment.setCreateBy(username);
        postComment.setUserId(userId);
        postComment.setUserName(username);
        postComment.setLikes(0L);
        postComment.setStatus("1"); // 默认正常状态
        
        // 如果父评论ID为空，设置为0（顶级评论）
        if (postComment.getParentId() == null) {
            postComment.setParentId(0L);
        }
        
        int result = postCommentService.insertPostComment(postComment);
        
        if (result > 0) {
            // 更新帖子评论数
            postService.updatePostComments(postComment.getPostId(), 1);
            
            // 返回新添加的评论，包括评论ID
            return success(postCommentService.selectPostCommentById(postComment.getCommentId()));
        } else {
            return error("添加评论失败");
        }
    }
    
    /**
     * 收藏/取消收藏帖子
     */
    @PostMapping("/favorite/{postId}")
    public AjaxResult toggleFavorite(@PathVariable Long postId, @RequestParam(required = false, defaultValue = "true") boolean favorite)
    {
        Long userId = null;
        try {
            userId = getUserId();
        } catch (Exception e) {
            // 用户未登录
            return error("请先登录");
        }
        
        if (userId == null) {
            return error("请先登录");
        }
        
        // 检查帖子是否存在
        Post post = postService.selectPostById(postId);
        if (post == null || !"1".equals(post.getStatus())) {
            return error("帖子不存在或已下架");
        }
        
        // 检查是否已收藏
        PostFavorite exist = postFavoriteService.selectPostFavoriteByPostIdAndUserId(postId, userId);
        
        Map<String, Object> resultData = new HashMap<>();
        resultData.put("favorited", favorite);
        
        if (favorite) {
            // 收藏操作
            if (exist != null) {
                // 已存在记录，检查状态
                if ("0".equals(exist.getStatus())) {
                    return AjaxResult.success("已收藏该帖子");
                } else {
                    // 取消收藏状态，改为收藏状态
                    exist.setStatus("0");
                    exist.setUpdateBy(getUsername());
                    postFavoriteService.updatePostFavorite(exist);
                    
                    // 更新帖子收藏数量
                    postService.updatePostFavorites(postId, 1);
                    
                    return AjaxResult.success("收藏成功");
                }
            } else {
                // 创建收藏记录
                PostFavorite favoriteRecord = new PostFavorite();
                favoriteRecord.setPostId(postId);
                favoriteRecord.setUserId(userId);
                favoriteRecord.setPostTitle(post.getTitle());
                favoriteRecord.setPostCover(post.getCoverImage());
                favoriteRecord.setStatus("0");
                favoriteRecord.setCreateBy(getUsername());
                postFavoriteService.insertPostFavorite(favoriteRecord);
                
                // 更新帖子收藏数量
                postService.updatePostFavorites(postId, 1);
                
                return AjaxResult.success("收藏成功");
            }
        } else {
            // 取消收藏操作
            if (exist != null && "0".equals(exist.getStatus())) {
                // 更新为取消收藏状态
                exist.setStatus("1");
                exist.setUpdateBy(getUsername());
                postFavoriteService.updatePostFavorite(exist);
                
                // 更新帖子收藏数量
                postService.updatePostFavorites(postId, -1);
                
                return AjaxResult.success("取消收藏成功");
            } else {
                return AjaxResult.success("未收藏该帖子");
            }
        }
    }
    
    /**
     * 获取热门标签
     */
    @Anonymous
    @GetMapping("/tags/hot")
    public AjaxResult getHotTags(@RequestParam(required = false, defaultValue = "10") int limit)
    {
        // 获取帖子数量最多的标签
        PostTag query = new PostTag();
        query.setStatus("0"); // 正常状态
        
        // 使用startPage方法
        startPage();
        
        // 手动限制结果数量
        List<PostTag> allTags = postTagService.selectPostTagList(query);
        List<PostTag> sortedTags = allTags.stream()
                .sorted((a, b) -> b.getPostCount().compareTo(a.getPostCount()))
                .limit(limit)
                .collect(java.util.stream.Collectors.toList());
        
        return AjaxResult.success(sortedTags);
    }
    
    /**
     * 根据标签获取帖子
     */
    @Anonymous
    @GetMapping("/bytag/{tagId}")
    public TableDataInfo getPostsByTag(@PathVariable Long tagId)
    {
        startPage();
        // 查询该标签的所有帖子
        // 假设已经实现了这个方法
        // 实际情况下，可能需要通过关联表查询
        PostTag tag = postTagService.selectPostTagById(tagId);
        
        if (tag == null) {
            return getDataTable(new ArrayList<>());
        }
        
        Post query = new Post();
        query.setTags(tag.getTagName());
        query.setStatus("1"); // 只查询正常状态的帖子
        List<Post> posts = postService.selectPostList(query);
        
        return getDataTable(posts);
    }

    /**
     * 发布新帖子
     */
    @PostMapping("/publish")
    public AjaxResult publishPost(@RequestBody Post post)
    {
        // 检查用户是否登录
        Long userId = null;
        String username = null;
        try {
            userId = getUserId();
            username = getUsername();
        } catch (Exception e) {
            return error("请先登录后再发布帖子");
        }
        
        if (userId == null) {
            return error("请先登录后再发布帖子");
        }
        
        // 验证必填字段
        if (post.getTitle() == null || post.getTitle().trim().isEmpty()) {
            return error("帖子标题不能为空");
        }
        
        if (post.getContent() == null || post.getContent().trim().isEmpty()) {
            return error("帖子内容不能为空");
        }
        
        if (post.getSummary() == null || post.getSummary().trim().isEmpty()) {
            return error("帖子摘要不能为空");
        }
        
        // 设置帖子基本信息
        post.setAuthorId(userId);
        post.setAuthor(username);
        post.setViews(0L);
        post.setComments(0L);
        post.setFavorites(0L);
        post.setLikes(0L);
        post.setStatus("0"); // 默认待审核状态
        post.setCreateBy(username);
        
        // 自动设置发布时间
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        post.setPublishTime(sdf.format(new Date()));
        
        // 如果没有设置封面图，使用默认图片
        if (post.getCoverImage() == null || post.getCoverImage().trim().isEmpty()) {
            post.setCoverImage("/images/default-cover.jpg");
        }
        
        // 处理标签
        if (post.getTags() != null && !post.getTags().trim().isEmpty()) {
            // 这里可以处理标签逻辑，比如创建标签关联关系
            // 暂时直接保存标签字符串
        }
        
        int result = postService.insertPost(post);
        
        if (result > 0) {
            return success("帖子发布成功，等待管理员审核");
        } else {
            return error("帖子发布失败");
        }
    }
    
    /**
     * 获取当前用户的帖子列表
     */
    @GetMapping("/my-posts")
    public TableDataInfo getMyPosts(Post post,
                                   @RequestParam(value = "pageNum", defaultValue = "1") Integer pageNum,
                                   @RequestParam(value = "pageSize", defaultValue = "10") Integer pageSize)
    {
        // 检查用户是否登录
        Long userId = null;
        try {
            userId = getUserId();
        } catch (Exception e) {
            return getDataTable(new ArrayList<>());
        }
        
        if (userId == null) {
            return getDataTable(new ArrayList<>());
        }
        
        // 手动设置分页参数
        PageHelper.startPage(pageNum, pageSize);
        
        // 只查询当前用户的帖子
        post.setAuthorId(userId);
        List<Post> list = postService.selectPostList(post);
        return getDataTable(list);
    }
    
    /**
     * 获取当前用户的收藏列表
     */
    @GetMapping("/my-favorites")
    public TableDataInfo getMyFavorites(@RequestParam(value = "pageNum", defaultValue = "1") Integer pageNum,
                                       @RequestParam(value = "pageSize", defaultValue = "10") Integer pageSize)
    {
        // 检查用户是否登录
        Long userId = null;
        try {
            userId = getUserId();
        } catch (Exception e) {
            return getDataTable(new ArrayList<>());
        }
        
        if (userId == null) {
            return getDataTable(new ArrayList<>());
        }
        
        // 手动设置分页参数
        PageHelper.startPage(pageNum, pageSize);
        
        // 查询用户收藏的帖子
        PostFavorite favoriteQuery = new PostFavorite();
        favoriteQuery.setUserId(userId);
        favoriteQuery.setStatus("0"); // 只查询有效收藏
        List<PostFavorite> favorites = postFavoriteService.selectPostFavoriteList(favoriteQuery);
        
        // 获取收藏的帖子详情
        List<Post> favoritePosts = new ArrayList<>();
        for (PostFavorite favorite : favorites) {
            Post post = postService.selectPostById(favorite.getPostId());
            if (post != null && "1".equals(post.getStatus())) {
                favoritePosts.add(post);
            }
        }
        
        return getDataTable(favoritePosts);
    }
    
    /**
     * 更新帖子
     */
    @PutMapping("/update")
    public AjaxResult updatePost(@RequestBody Post post)
    {
        // 检查用户是否登录
        Long userId = null;
        String username = null;
        try {
            userId = getUserId();
            username = getUsername();
        } catch (Exception e) {
            return error("请先登录");
        }
        
        if (userId == null) {
            return error("请先登录");
        }
        
        if (post.getPostId() == null) {
            return error("帖子ID不能为空");
        }
        
        // 检查帖子是否存在且属于当前用户
        Post existingPost = postService.selectPostById(post.getPostId());
        if (existingPost == null) {
            return error("帖子不存在");
        }
        
        if (!userId.equals(existingPost.getAuthorId())) {
            return error("只能编辑自己的帖子");
        }
        
        // 验证必填字段
        if (post.getTitle() == null || post.getTitle().trim().isEmpty()) {
            return error("帖子标题不能为空");
        }
        
        if (post.getContent() == null || post.getContent().trim().isEmpty()) {
            return error("帖子内容不能为空");
        }
        
        if (post.getSummary() == null || post.getSummary().trim().isEmpty()) {
            return error("帖子摘要不能为空");
        }
        
        // 设置更新信息
        post.setUpdateBy(username);
        
        int result = postService.updatePost(post);
        
        if (result > 0) {
            return success("帖子更新成功");
        } else {
            return error("帖子更新失败");
        }
    }
    
    /**
     * 删除帖子
     */
    @DeleteMapping("/delete/{postId}")
    public AjaxResult deletePost(@PathVariable Long postId)
    {
        // 检查用户是否登录
        Long userId = null;
        try {
            userId = getUserId();
        } catch (Exception e) {
            return error("请先登录");
        }
        
        if (userId == null) {
            return error("请先登录");
        }
        
        // 检查帖子是否存在且属于当前用户
        Post existingPost = postService.selectPostById(postId);
        if (existingPost == null) {
            return error("帖子不存在");
        }
        
        if (!userId.equals(existingPost.getAuthorId())) {
            return error("只能删除自己的帖子");
        }
        
        try {
            // 级联删除相关数据
            // 1. 删除帖子评论
            postCommentService.deletePostCommentByPostId(postId);
            
            // 2. 删除帖子收藏记录
            postFavoriteService.deletePostFavoriteByPostId(postId);
            
            // 3. 删除帖子本身
        int result = postService.deletePostById(postId);
        
        if (result > 0) {
            return success("帖子删除成功");
        } else {
            return error("帖子删除失败");
            }
        } catch (Exception e) {
            return error("删除失败：" + e.getMessage());
        }
    }
    
    /**
     * 获取可用标签列表（用于下拉选择）
     */
    @Anonymous
    @GetMapping("/tags/available")
    public AjaxResult getAvailableTags()
    {
        PostTag query = new PostTag();
        query.setStatus("0"); // 正常状态
        List<PostTag> tags = postTagService.selectPostTagList(query);
        return success(tags);
    }
    
    /**
     * 点赞/取消点赞帖子
     */
    @PostMapping("/like/{postId}")
    public AjaxResult toggleLike(@PathVariable Long postId, @RequestParam(required = false, defaultValue = "true") boolean like)
    {
        Long userId = null;
        try {
            userId = getUserId();
        } catch (Exception e) {
            return error("请先登录");
        }
        
        if (userId == null) {
            return error("请先登录");
        }
        
        // 检查帖子是否存在
        Post post = postService.selectPostById(postId);
        if (post == null || !"1".equals(post.getStatus())) {
            return error("帖子不存在或已下架");
        }
        
        // 这里简化处理，直接更新点赞数
        // 实际应用中应该记录用户点赞状态，防止重复点赞
        if (like) {
            postService.updatePostLikes(postId, 1);
            return success("点赞成功");
        } else {
            postService.updatePostLikes(postId, -1);
            return success("取消点赞成功");
        }
    }
    
    /**
     * 点赞/取消点赞评论
     */
    @PostMapping("/comment/like/{commentId}")
    public AjaxResult toggleCommentLike(@PathVariable Long commentId, @RequestParam(required = false, defaultValue = "true") boolean like)
    {
        Long userId = null;
        try {
            userId = getUserId();
        } catch (Exception e) {
            return error("请先登录");
        }
        
        if (userId == null) {
            return error("请先登录");
        }
        
        // 检查评论是否存在
        PostComment comment = postCommentService.selectPostCommentById(commentId);
        if (comment == null || !"1".equals(comment.getStatus())) {
            return error("评论不存在或已删除");
        }
        
        // 这里简化处理，直接更新点赞数
        // 实际应用中应该记录用户点赞状态，防止重复点赞
        if (like) {
            postCommentService.updateCommentLikes(commentId, 1);
            return success("点赞成功");
        } else {
            postCommentService.updateCommentLikes(commentId, -1);
            return success("取消点赞成功");
        }
    }
    
    /**
     * 上传图片
     */
    @PostMapping("/upload")
    public AjaxResult uploadImage(@RequestParam("file") MultipartFile file)
    {
        try {
            if (file.isEmpty()) {
                return error("上传文件不能为空");
            }
            
            // 检查文件类型
            String originalFilename = file.getOriginalFilename();
            if (originalFilename == null) {
                return error("文件名不能为空");
            }
            
            String extension = originalFilename.substring(originalFilename.lastIndexOf(".")).toLowerCase();
            if (!extension.matches("(?i)\\.(jpg|jpeg|png|gif|bmp|webp)$")) {
                return error("只支持jpg、jpeg、png、gif、bmp、webp格式的图片文件");
            }
            
            // 检查文件大小（限制为5MB）
            if (file.getSize() > 5 * 1024 * 1024) {
                return error("文件大小不能超过5MB");
            }
            
            // 生成唯一文件名
            String fileName = System.currentTimeMillis() + "_" + 
                             (int)(Math.random() * 1000) + extension;
            
            // 使用配置文件中的相对路径
            String frontendImagePath = com.ruoyi.common.config.RuoYiConfig.getFrontendImagePath();
            if (frontendImagePath == null) {
                return error("前端图片路径未配置");
            }
            
            // 获取项目根目录的绝对路径
            String projectRoot = System.getProperty("user.dir");
            String uploadDir = Paths.get(projectRoot, frontendImagePath).toString();
            
            File uploadDirFile = new File(uploadDir);
            if (!uploadDirFile.exists()) {
                uploadDirFile.mkdirs();
            }
            
            // 保存文件
            Path filePath = Paths.get(uploadDir, fileName);
            Files.copy(file.getInputStream(), filePath);
            
            // 返回访问URL - 前端可以直接通过/images/访问
            String imageUrl = "/images/" + fileName;
            
            Map<String, String> result = new HashMap<>();
            result.put("url", imageUrl);
            result.put("filename", fileName);
            result.put("originalName", originalFilename);
            result.put("size", String.valueOf(file.getSize()));
            
            return success(result);
        } catch (IOException e) {
            return error("上传失败：" + e.getMessage());
        } catch (Exception e) {
            return error("上传失败：" + e.getMessage());
        }
    }

    /**
     * 删除图片
     */
    @DeleteMapping("/upload")
    public AjaxResult deleteImage(@RequestParam("filename") String filename)
    {
        try {
            if (filename == null || filename.trim().isEmpty()) {
                return error("文件名不能为空");
            }
            
            // 使用配置文件中的相对路径
            String frontendImagePath = com.ruoyi.common.config.RuoYiConfig.getFrontendImagePath();
            if (frontendImagePath == null) {
                return error("前端图片路径未配置");
            }
            
            // 获取项目根目录的绝对路径
            String projectRoot = System.getProperty("user.dir");
            String uploadDir = Paths.get(projectRoot, frontendImagePath).toString();
            Path filePath = Paths.get(uploadDir, filename);
            
            // 检查文件是否存在
            if (Files.exists(filePath)) {
                Files.delete(filePath);
                return success("图片删除成功");
            } else {
                return error("文件不存在");
            }
        } catch (IOException e) {
            return error("删除失败：" + e.getMessage());
        } catch (Exception e) {
            return error("删除失败：" + e.getMessage());
        }
    }

    /**
     * 获取推广计划列表
     */
    @Anonymous
    @GetMapping("/promotion-plans")
    public AjaxResult getPromotionPlans()
    {
        // 这里应该从数据库查询推广计划，暂时返回硬编码数据
        List<Map<String, Object>> plans = new ArrayList<>();
        
        Map<String, Object> plan1 = new HashMap<>();
        plan1.put("id", 1);
        plan1.put("name", "首页推荐");
        plan1.put("description", "帖子显示在首页推荐位置");
        plan1.put("price", 50.00);
        plan1.put("duration", 7);
        plans.add(plan1);
        
        Map<String, Object> plan2 = new HashMap<>();
        plan2.put("id", 2);
        plan2.put("name", "置顶推广");
        plan2.put("description", "帖子在分类页面置顶显示");
        plan2.put("price", 30.00);
        plan2.put("duration", 3);
        plans.add(plan2);
        
        Map<String, Object> plan3 = new HashMap<>();
        plan3.put("id", 3);
        plan3.put("name", "热门推荐");
        plan3.put("description", "帖子显示在热门推荐区域");
        plan3.put("price", 80.00);
        plan3.put("duration", 14);
        plans.add(plan3);
        
        Map<String, Object> plan4 = new HashMap<>();
        plan4.put("id", 4);
        plan4.put("name", "限时闪推");
        plan4.put("description", "特定时间段内高优显示");
        plan4.put("price", 20.00);
        plan4.put("duration", 1);
        plans.add(plan4);
        
        Map<String, Object> plan5 = new HashMap<>();
        plan5.put("id", 5);
        plan5.put("name", "分类页首条");
        plan5.put("description", "在所属分类列表第一条显示（非置顶样式）");
        plan5.put("price", 40.00);
        plan5.put("duration", 5);
        plans.add(plan5);
        
        return success(plans);
    }

    /**
     * 创建支付记录
     */
    @PostMapping("/payment")
    public AjaxResult createPayment(@RequestBody PaymentRequest request)
    {
        // 检查用户是否登录
        Long userId = null;
        try {
            userId = getUserId();
        } catch (Exception e) {
            return error("请先登录");
        }
        
        if (userId == null) {
            return error("请先登录");
        }
        
        // 验证参数
        if (request.getPostId() == null || request.getPlanId() == null || request.getAmount() == null) {
            return error("参数不完整");
        }
        
        // 检查帖子是否存在且属于当前用户
        Post post = postService.selectPostById(request.getPostId());
        if (post == null) {
            return error("帖子不存在");
        }
        
        if (!userId.equals(post.getAuthorId())) {
            return error("只能为自己的帖子购买推广");
        }
        
        // 创建支付记录
        PostPayment payment = new PostPayment();
        payment.setPostId(request.getPostId());
        payment.setPlanId(request.getPlanId());
        payment.setUserId(userId);
        payment.setAmount(BigDecimal.valueOf(request.getAmount()));
        payment.setPaymentStatus("pending"); // 待支付状态
        payment.setPaymentTime(new Date());
        
        int result = postPaymentMapper.insertPostPayment(payment);
        
        if (result > 0) {
            return success(payment);
        } else {
            return error("创建支付记录失败");
        }
    }

    /**
     * 检查帖子推广状态
     */
    @GetMapping("/promotion-status/{postId}")
    public AjaxResult getPromotionStatus(@PathVariable Long postId)
    {
        // 检查用户是否登录
        Long userId = null;
        try {
            userId = getUserId();
        } catch (Exception e) {
            return error("请先登录");
        }
        
        if (userId == null) {
            return error("请先登录");
        }
        
        // 检查帖子是否存在且属于当前用户
        Post post = postService.selectPostById(postId);
        if (post == null) {
            return error("帖子不存在");
        }
        
        if (!userId.equals(post.getAuthorId())) {
            return error("只能查看自己的帖子推广状态");
        }
        
        Map<String, Object> result = new HashMap<>();
        result.put("hasPromotion", post.getPromotionPlanId() != null);
        result.put("promotionPlanId", post.getPromotionPlanId());
        
        // 查询最新的支付记录
        if (post.getPromotionPlanId() != null) {
            PostPayment latestPayment = postPaymentMapper.selectLatestPaymentByPostId(postId);
            result.put("latestPayment", latestPayment);
        }
        
        return success(result);
    }

    /**
     * 确认支付成功
     */
    @PostMapping("/payment/confirm/{paymentId}")
    public AjaxResult confirmPayment(@PathVariable Long paymentId)
    {
        // 检查用户是否登录
        Long userId = null;
        try {
            userId = getUserId();
        } catch (Exception e) {
            return error("请先登录");
        }
        
        if (userId == null) {
            return error("请先登录");
        }
        
        // 查询支付记录
        PostPayment payment = postPaymentMapper.selectPostPaymentById(paymentId);
        if (payment == null) {
            return error("支付记录不存在");
        }
        
        if (!userId.equals(payment.getUserId())) {
            return error("只能确认自己的支付记录");
        }
        
        if (!"pending".equals(payment.getPaymentStatus())) {
            return error("支付记录状态异常");
        }
        
        // 更新支付状态为成功
        payment.setPaymentStatus("paid");
        payment.setPaymentTime(new Date());
        postPaymentMapper.updatePostPayment(payment);
        
        // 更新帖子的推广计划ID
        Post updatePost = new Post();
        updatePost.setPostId(payment.getPostId());
        updatePost.setPromotionPlanId(payment.getPlanId());
        updatePost.setUpdateBy(getUsername());
        
        int result = postService.updatePost(updatePost);
        
        if (result > 0) {
            return success("支付成功，推广已生效");
        } else {
            return error("支付确认失败");
        }
    }

    /**
     * 取消支付
     */
    @PostMapping("/payment/cancel/{paymentId}")
    public AjaxResult cancelPayment(@PathVariable Long paymentId)
    {
        // 检查用户是否登录
        Long userId = null;
        try {
            userId = getUserId();
        } catch (Exception e) {
            return error("请先登录");
        }
        
        if (userId == null) {
            return error("请先登录");
        }
        
        // 查询支付记录
        PostPayment payment = postPaymentMapper.selectPostPaymentById(paymentId);
        if (payment == null) {
            return error("支付记录不存在");
        }
        
        if (!userId.equals(payment.getUserId())) {
            return error("只能取消自己的支付记录");
        }
        
        if (!"pending".equals(payment.getPaymentStatus())) {
            return error("支付记录状态异常");
        }
        
        // 更新支付状态为失败
        payment.setPaymentStatus("failed");
        postPaymentMapper.updatePostPayment(payment);
        
        return success("支付已取消");
    }

    /**
     * 支付请求类
     */
    public static class PaymentRequest {
        private Long postId;
        private Long planId;
        private Double amount;

        public Long getPostId() {
            return postId;
        }

        public void setPostId(Long postId) {
            this.postId = postId;
        }

        public Long getPlanId() {
            return planId;
        }

        public void setPlanId(Long planId) {
            this.planId = planId;
        }

        public Double getAmount() {
            return amount;
        }

        public void setAmount(Double amount) {
            this.amount = amount;
        }
    }

    /**
     * 举报帖子
     */
    @PostMapping("/report/{postId}")
    public AjaxResult reportPost(@PathVariable Long postId, @RequestBody ReportRequest request)
    {
        // 检查用户是否登录
        Long userId = null;
        String username = null;
        try {
            userId = getUserId();
            username = getUsername();
        } catch (Exception e) {
            return error("请先登录");
        }
        
        if (userId == null) {
            return error("请先登录");
        }
        
        // 检查帖子是否存在
        Post post = postService.selectPostById(postId);
        if (post == null || !"1".equals(post.getStatus())) {
            return error("帖子不存在或已下架");
        }
        
        // 检查举报理由
        if (request.getReason() == null || request.getReason().trim().isEmpty()) {
            return error("举报理由不能为空");
        }
        
        // 创建举报记录
        PostReport report = new PostReport();
        report.setPostId(postId);
        report.setPostTitle(post.getTitle());
        report.setReportUserId(userId);
        report.setReportUserName(username);
        report.setReportReason(request.getReason());
        report.setStatus("0"); // 待处理状态
        report.setCreateBy(username);
        
        int result = postReportMapper.insertPostReport(report);
        
        if (result > 0) {
            return success("举报提交成功，我们会尽快处理");
        } else {
            return error("举报提交失败");
        }
    }

    /**
     * 举报请求类
     */
    public static class ReportRequest {
        private String reason;

        public String getReason() {
            return reason;
        }

        public void setReason(String reason) {
            this.reason = reason;
        }
    }
} 