package com.ruoyi.web.Server.BT;

import cn.hutool.core.io.FileUtil;
import com.ruoyi.web.controller.common.exception.RocketPTException;
import com.ruoyi.web.Tool.BT.TorrentUtils;
import jakarta.annotation.PostConstruct;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.Setter;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;
import org.springframework.web.multipart.MultipartFile;

import java.io.File;
import java.io.InputStream;


/**
 * 本地种子存储
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class LocalTorrentStorageService implements TorrentStorageService {

    /**
     * 使用绝对路径
     */
    @Getter
    @Setter
    boolean useAbsolutePath = false;

    @Getter
    @Setter
    String absolutePath = "/torrent/";

    @Getter
    @Setter
    String relativePath = "/torrent/";


    String defaultDir = TorrentUtils.getDefaultDir();


    @PostConstruct
    @Override
    @SneakyThrows
    public void init() {
        // 1. 初始化绝对路径（基于相对路径自动生成）
        if (useAbsolutePath) {
            // 若需绝对路径，可基于类路径或系统目录生成
            absolutePath = new File(relativePath).getAbsolutePath();
        } else {
            absolutePath = relativePath; // 直接使用相对路径（由运行环境决定解析方式）
        }

        // 2. 确保路径存在（自动创建目录）
        FileUtil.mkdir(absolutePath);
    }


    @Override
    public void save(Integer id, byte[] torrent) {
        String path = getFilePath(id);

        FileUtil.writeBytes(torrent, path);
    }

    public String getPath() {
        if (useAbsolutePath) {
            return absolutePath;
        }

        return defaultDir + relativePath;
    }

    public String getFilePath(int id) {
        String path = getPath() + id + ".torrent";

        return path;
    }

    @Override
    @SneakyThrows
    public void store(Integer id, MultipartFile file) {
        String filename = StringUtils.cleanPath(file.getOriginalFilename());
        // validation here for .torrent file
        if (!filename.endsWith(".torrent")) {
            throw new RocketPTException("Invalid file type. Only .torrent files are allowed");
        }

        if (file.isEmpty()) {
            throw new RocketPTException("Failed to store empty file " + filename);
        }
        if (filename.contains("..")) {
            // This is a security check
            throw new RocketPTException("Cannot store file with relative path outside current" +
                    " directory "
                    + filename);
        }

        try (InputStream inputStream = file.getInputStream()) {
            FileUtil.writeFromStream(inputStream, getFilePath(id));
        }

    }

    @Override
    public byte[] read(Integer id) {
        String path = getFilePath(id);

        return FileUtil.readBytes(path);
    }

    @Override
    public void delete(Integer id) {
        String path = getFilePath(id);
        FileUtil.del(path);
    }

    @Override
    public InputStream load(Integer id) {
        String path = getFilePath(id);
        return FileUtil.getInputStream(path);

    }

}
