package com.ruoyi.web.controller.post.controller;

import java.util.List;
import jakarta.servlet.http.HttpServletResponse;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import com.ruoyi.common.annotation.Log;
import com.ruoyi.common.core.controller.BaseController;
import com.ruoyi.common.core.domain.AjaxResult;
import com.ruoyi.common.core.page.TableDataInfo;
import com.ruoyi.common.enums.BusinessType;
import com.ruoyi.common.utils.poi.ExcelUtil;
import com.ruoyi.web.controller.post.domain.PostTag;
import com.ruoyi.web.controller.post.service.IPostTagService;

/**
 * 帖子标签管理
 * 
 * @author thunderhub
 */
@RestController
@RequestMapping("/post/tag")
public class PostTagController extends BaseController
{
    @Autowired
    private IPostTagService postTagService;

    /**
     * 获取帖子标签列表
     */
    @PreAuthorize("@ss.hasPermi('post:tag:list')")
    @GetMapping("/list")
    public TableDataInfo list(PostTag postTag)
    {
        startPage();
        List<PostTag> list = postTagService.selectPostTagList(postTag);
        return getDataTable(list);
    }
    
    /**
     * 导出帖子标签列表
     */
    @PreAuthorize("@ss.hasPermi('post:tag:export')")
    @Log(title = "帖子标签", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(HttpServletResponse response, PostTag postTag)
    {
        List<PostTag> list = postTagService.selectPostTagList(postTag);
        ExcelUtil<PostTag> util = new ExcelUtil<PostTag>(PostTag.class);
        util.exportExcel(response, list, "标签数据");
    }

    /**
     * 获取帖子标签详细信息
     */
    @PreAuthorize("@ss.hasPermi('post:tag:query')")
    @GetMapping(value = "/{tagId}")
    public AjaxResult getInfo(@PathVariable Long tagId)
    {
        return success(postTagService.selectPostTagById(tagId));
    }

    /**
     * 新增帖子标签
     */
    @PreAuthorize("@ss.hasPermi('post:tag:add')")
    @Log(title = "帖子标签", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@Validated @RequestBody PostTag postTag)
    {
        if (!postTagService.checkTagNameUnique(postTag))
        {
            return error("新增标签'" + postTag.getTagName() + "'失败，标签名称已存在");
        }
        postTag.setCreateBy(getUsername());
        postTag.setPostCount(0L);
        return toAjax(postTagService.insertPostTag(postTag));
    }

    /**
     * 修改帖子标签
     */
    @PreAuthorize("@ss.hasPermi('post:tag:edit')")
    @Log(title = "帖子标签", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@Validated @RequestBody PostTag postTag)
    {
        if (!postTagService.checkTagNameUnique(postTag))
        {
            return error("修改标签'" + postTag.getTagName() + "'失败，标签名称已存在");
        }
        postTag.setUpdateBy(getUsername());
        return toAjax(postTagService.updatePostTag(postTag));
    }

    /**
     * 删除帖子标签
     */
    @PreAuthorize("@ss.hasPermi('post:tag:remove')")
    @Log(title = "帖子标签", businessType = BusinessType.DELETE)
    @DeleteMapping("/{tagIds}")
    public AjaxResult remove(@PathVariable Long[] tagIds)
    {
        return toAjax(postTagService.deletePostTagByIds(tagIds));
    }
    
    /**
     * 获取帖子的标签列表
     */
    @PreAuthorize("@ss.hasPermi('post:tag:list')")
    @GetMapping("/list/{postId}")
    public AjaxResult listByPostId(@PathVariable Long postId)
    {
        List<PostTag> tags = postTagService.selectPostTagsByPostId(postId);
        return success(tags);
    }
    
    /**
     * 获取所有可用标签（用于下拉选择）
     */
    @GetMapping("/optionselect")
    public AjaxResult optionselect()
    {
        PostTag query = new PostTag();
        query.setStatus("0"); // 正常状态
        List<PostTag> tags = postTagService.selectPostTagList(query);
        return success(tags);
    }
} 