package com.ruoyi.web.controller.post;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.ruoyi.common.core.domain.entity.SysUser;
import com.ruoyi.common.core.domain.model.LoginUser;
import com.ruoyi.web.controller.post.controller.PostCenterController;
import com.ruoyi.web.controller.post.domain.*;
import com.ruoyi.web.controller.post.service.*;
import com.ruoyi.web.controller.post.mapper.*;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockedStatic;
import org.mockito.junit.jupiter.MockitoExtension;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.http.MediaType;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.test.context.support.WithMockUser;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;
import org.springframework.web.multipart.MultipartFile;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

import static org.mockito.ArgumentMatchers.*;
import static org.mockito.Mockito.*;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.*;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.*;
import org.springframework.boot.test.context.SpringBootTestContextBootstrapper;
import org.springframework.test.context.BootstrapWith;
/**
 * PostCenterController 单元测试
 * 
 * 测试覆盖所有帖子中心相关的API接口
 * 包括：帖子列表、详情、评论、收藏、点赞、发布、上传等功能
 */
@ExtendWith(MockitoExtension.class)
@WithMockUser(username = "testUser", authorities = {"admin"})
class PostCenterControllerTest {

    @Mock
    private IPostService postService;

    @Mock
    private IPostTagService postTagService;

    @Mock
    private IPostCommentService postCommentService;

    @Mock
    private IPostFavoriteService postFavoriteService;

    @Mock
    private PostReportMapper postReportMapper;

    @Mock
    private PostPaymentMapper postPaymentMapper;

    @InjectMocks
    private PostCenterController postCenterController;

    private MockMvc mockMvc;
    private ObjectMapper objectMapper;

    @BeforeEach
    void setUp() {
        // 创建模拟的安全上下文
        SysUser sysUser = new SysUser();
        sysUser.setUserId(1L);
        sysUser.setUserName("testUser");
        
        LoginUser loginUser = new LoginUser(1L, 1L, sysUser, new HashSet<>());
        
        UsernamePasswordAuthenticationToken authToken = 
            new UsernamePasswordAuthenticationToken(loginUser, null, loginUser.getAuthorities());
        SecurityContextHolder.getContext().setAuthentication(authToken);
        
        mockMvc = MockMvcBuilders.standaloneSetup(postCenterController).build();
        objectMapper = new ObjectMapper();
    }

    /**
     * 测试获取帖子列表
     */
    @Test
    void testGetPostList() throws Exception {
        List<Post> posts = new ArrayList<>();
        Post post = new Post();
        post.setPostId(1L);
        post.setTitle("测试帖子");
        posts.add(post);

        when(postService.selectPostList(any(Post.class))).thenReturn(posts);

        mockMvc.perform(get("/post-center/list"))
                .andExpect(status().isOk());

        verify(postService, times(1)).selectPostList(any(Post.class));
    }

    /**
     * 测试获取帖子详情
     */
    @Test
    void testGetPostDetail() throws Exception {
        Post post = new Post();
        post.setPostId(1L);
        post.setTitle("测试帖子");
        post.setStatus("1");

        when(postService.updatePostViews(1L)).thenReturn(1);
        when(postService.selectPostById(1L)).thenReturn(post);
        when(postTagService.selectPostTagsByPostId(1L)).thenReturn(new ArrayList<>());
        when(postCommentService.selectCommentsByPostId(eq(1L), eq(10))).thenReturn(new ArrayList<>());
        when(postService.getRecommendedPosts(any(), any(), eq(1L), eq(9))).thenReturn(new ArrayList<>());

        mockMvc.perform(get("/post-center/{postId}", 1L))
                .andExpect(status().isOk());

        verify(postService, times(1)).selectPostById(1L);
        verify(postService, times(1)).updatePostViews(1L);
    }

    /**
     * 测试发布帖子
     */
    @Test
    void testPublishPost() throws Exception {
        when(postService.insertPost(any(Post.class))).thenReturn(1);

        String postData = "{\"title\":\"新帖子\",\"content\":\"帖子内容\",\"summary\":\"帖子摘要\"}";

        mockMvc.perform(post("/post-center/publish")
                .contentType(MediaType.APPLICATION_JSON)
                .content(postData))
                .andExpect(status().isOk());

        verify(postService, times(1)).insertPost(any(Post.class));
    }

    /**
     * 测试更新帖子
     */
    @Test
    void testUpdatePost() throws Exception {
        Post existingPost = new Post();
        existingPost.setPostId(1L);
        existingPost.setAuthorId(1L);

        when(postService.selectPostById(1L)).thenReturn(existingPost);
        when(postService.updatePost(any(Post.class))).thenReturn(1);

        String postData = "{\"postId\":1,\"title\":\"更新帖子\",\"content\":\"更新内容\",\"summary\":\"更新摘要\"}";

        mockMvc.perform(put("/post-center/update")
                .contentType(MediaType.APPLICATION_JSON)
                .content(postData))
                .andExpect(status().isOk());

        verify(postService, times(1)).updatePost(any(Post.class));
    }

    /**
     * 测试删除帖子
     */
    @Test
    void testDeletePost() throws Exception {
        Post existingPost = new Post();
        existingPost.setPostId(1L);
        existingPost.setAuthorId(1L);

        when(postService.selectPostById(1L)).thenReturn(existingPost);
        when(postCommentService.deletePostCommentByPostId(1L)).thenReturn(1);
        when(postFavoriteService.deletePostFavoriteByPostId(1L)).thenReturn(1);
        when(postService.deletePostById(1L)).thenReturn(1);

        mockMvc.perform(delete("/post-center/delete/{postId}", 1L))
                .andExpect(status().isOk());

        verify(postService, times(1)).deletePostById(1L);
    }

    /**
     * 测试添加评论
     */
    @Test
    void testAddComment() throws Exception {
        PostComment comment = new PostComment();
        comment.setPostId(1L);
        comment.setContent("测试评论");
        comment.setParentId(0L);

        when(postCommentService.selectPostCommentById(any())).thenReturn(null);
        when(postCommentService.insertPostComment(any(PostComment.class))).thenReturn(1);
        when(postService.updatePostComments(eq(1L), anyInt())).thenReturn(1);

        String commentData = objectMapper.writeValueAsString(comment);

        mockMvc.perform(post("/post-center/comment")
                .contentType(MediaType.APPLICATION_JSON)
                .content(commentData))
                .andExpect(status().isOk());

        verify(postCommentService, times(1)).insertPostComment(any(PostComment.class));
    }

    /**
     * 测试获取评论列表（实际是测试帖子列表）
     */
    @Test
    void testGetComments() throws Exception {
        List<Post> posts = new ArrayList<>();
        Post post = new Post();
        post.setPostId(1L);
        post.setTitle("测试帖子");
        posts.add(post);

        when(postService.selectPostList(any(Post.class))).thenReturn(posts);

        // 测试帖子列表接口
        mockMvc.perform(get("/post-center/list")
                .param("postId", "1")
                .param("pageSize", "10"))
                .andExpect(status().isOk());

        verify(postService, times(1)).selectPostList(any(Post.class));
    }

    /**
     * 测试收藏帖子
     */
    @Test
    void testToggleFavorite() throws Exception {
        Post post = new Post();
        post.setPostId(1L);
        post.setStatus("1");
        post.setTitle("测试帖子");

        when(postService.selectPostById(1L)).thenReturn(post);
        when(postFavoriteService.selectPostFavoriteByPostIdAndUserId(anyLong(), anyLong())).thenReturn(null);
        when(postFavoriteService.insertPostFavorite(any())).thenReturn(1);
        when(postService.updatePostFavorites(anyLong(), anyInt())).thenReturn(1);

        mockMvc.perform(post("/post-center/favorite/{postId}", 1L))
                .andExpect(status().isOk());

        verify(postFavoriteService, times(1)).insertPostFavorite(any());
    }

    /**
     * 测试点赞帖子
     */
    @Test
    void testToggleLike() throws Exception {
        Post post = new Post();
        post.setPostId(1L);
        post.setStatus("1");

        when(postService.selectPostById(1L)).thenReturn(post);
        when(postService.updatePostLikes(anyLong(), anyInt())).thenReturn(1);

        mockMvc.perform(post("/post-center/like/{postId}", 1L))
                .andExpect(status().isOk());

        verify(postService, times(1)).updatePostLikes(eq(1L), anyInt());
    }

    /**
     * 测试上传图片
     */
    @Test
    void testUploadImage() throws Exception {
        mockMvc.perform(multipart("/post-center/upload")
                .file("file", "test image content".getBytes())
                .param("filename", "test.jpg"))
                .andExpect(status().isOk());
    }

    /**
     * 测试获取我的帖子
     */
    @Test
    void testGetMyPosts() throws Exception {
        List<Post> posts = new ArrayList<>();
        when(postService.selectPostList(any())).thenReturn(posts);

        mockMvc.perform(get("/post-center/my-posts"))
                .andExpect(status().isOk());

        verify(postService, times(1)).selectPostList(any());
    }

    /**
     * 测试获取我的收藏
     */
    @Test
    void testGetMyFavorites() throws Exception {
        List<PostFavorite> favorites = new ArrayList<>();
        when(postFavoriteService.selectPostFavoriteList(any())).thenReturn(favorites);

        mockMvc.perform(get("/post-center/my-favorites"))
                .andExpect(status().isOk());

        verify(postFavoriteService, times(1)).selectPostFavoriteList(any());
    }

    /**
     * 测试举报帖子
     */
    @Test
    void testReportPost() throws Exception {
        Post post = new Post();
        post.setPostId(1L);
        post.setStatus("1");
        post.setTitle("测试帖子");

        when(postService.selectPostById(1L)).thenReturn(post);
        when(postReportMapper.insertPostReport(any())).thenReturn(1);

        String reportData = "{\"reason\":\"垃圾内容\"}";

        mockMvc.perform(post("/post-center/report/{postId}", 1L)
                .contentType(MediaType.APPLICATION_JSON)
                .content(reportData))
                .andExpect(status().isOk());

        verify(postReportMapper, times(1)).insertPostReport(any());
    }

    /**
     * 测试获取热门标签
     */
    @Test
    void testGetHotTags() throws Exception {
        List<PostTag> tags = new ArrayList<>();
        when(postTagService.selectPostTagList(any())).thenReturn(tags);

        mockMvc.perform(get("/post-center/tags/hot"))
                .andExpect(status().isOk());

        verify(postTagService, times(1)).selectPostTagList(any());
    }

    /**
     * 测试获取可用标签
     */
    @Test
    void testGetAvailableTags() throws Exception {
        List<PostTag> tags = new ArrayList<>();
        when(postTagService.selectPostTagList(any())).thenReturn(tags);

        mockMvc.perform(get("/post-center/tags/available"))
                .andExpect(status().isOk());

        verify(postTagService, times(1)).selectPostTagList(any());
    }

    /**
     * 测试点赞评论
     */
    @Test
    void testToggleCommentLike() throws Exception {
        PostComment comment = new PostComment();
        comment.setCommentId(1L);
        comment.setStatus("1");

        when(postCommentService.selectPostCommentById(1L)).thenReturn(comment);
        when(postCommentService.updateCommentLikes(eq(1L), anyInt())).thenReturn(1);

        mockMvc.perform(post("/post-center/comment/like/{commentId}", 1L))
                .andExpect(status().isOk());

        verify(postCommentService, times(1)).updateCommentLikes(eq(1L), anyInt());
    }

    /**
     * 测试获取帖子详情（帖子不存在）
     */
    @Test
    void testGetPostDetailNotFound() throws Exception {
        when(postService.updatePostViews(999L)).thenReturn(0);
        when(postService.selectPostById(999L)).thenReturn(null);

        mockMvc.perform(get("/post-center/{postId}", 999L))
                .andExpect(status().isOk());

        verify(postService, times(1)).selectPostById(999L);
    }

    /**
     * 测试发布空内容帖子
     */
    @Test
    void testPublishEmptyPost() throws Exception {
        String emptyPostData = "{\"title\":\"\",\"content\":\"\"}";

        mockMvc.perform(post("/post-center/publish")
                .contentType(MediaType.APPLICATION_JSON)
                .content(emptyPostData))
                .andExpect(status().isOk());
    }

    /**
     * 测试添加空评论
     */
    @Test
    void testAddEmptyComment() throws Exception {
        String emptyCommentData = "{\"postId\":1,\"content\":\"\"}";

        mockMvc.perform(post("/post-center/comment")
                .contentType(MediaType.APPLICATION_JSON)
                .content(emptyCommentData))
                .andExpect(status().isOk());
    }

    /**
     * 测试上传空文件
     */
    @Test
    void testUploadEmptyFile() throws Exception {
        mockMvc.perform(multipart("/post-center/upload")
                .file("file", new byte[0])
                .param("filename", ""))
                .andExpect(status().isOk());
    }

    /**
     * 测试获取推广计划
     */
    @Test
    void testGetPromotionPlans() throws Exception {
        mockMvc.perform(get("/post-center/promotion-plans"))
                .andExpect(status().isOk());
    }

    /**
     * 测试创建支付
     */
    @Test
    void testCreatePayment() throws Exception {
        Post post = new Post();
        post.setPostId(1L);
        post.setAuthorId(1L);

        when(postService.selectPostById(1L)).thenReturn(post);
        when(postPaymentMapper.insertPostPayment(any())).thenReturn(1);

        String paymentData = "{\"postId\":1,\"planId\":1,\"amount\":100}";

        mockMvc.perform(post("/post-center/payment")
                .contentType(MediaType.APPLICATION_JSON)
                .content(paymentData))
                .andExpect(status().isOk());

        verify(postPaymentMapper, times(1)).insertPostPayment(any());
    }

    /**
     * 测试获取推广状态
     */
    @Test
    void testGetPromotionStatus() throws Exception {
        Post post = new Post();
        post.setPostId(1L);
        post.setAuthorId(1L);

        when(postService.selectPostById(1L)).thenReturn(post);

        mockMvc.perform(get("/post-center/promotion-status/{postId}", 1L))
                .andExpect(status().isOk());
    }

    /**
     * 测试获取推广帖子
     */
    @Test
    void testGetPromotionPosts() throws Exception {
        List<Post> posts = new ArrayList<>();
        when(postService.selectPostList(any())).thenReturn(posts);

        mockMvc.perform(get("/post-center/promotion"))
                .andExpect(status().isOk());

        verify(postService, times(1)).selectPostList(any());
    }

    /**
     * 测试根据标签获取帖子
     */
    @Test
    void testGetPostsByTag() throws Exception {
        PostTag tag = new PostTag();
        tag.setTagId(1L);
        tag.setTagName("测试标签");
        
        List<Post> posts = new ArrayList<>();
        when(postTagService.selectPostTagById(1L)).thenReturn(tag);
        when(postService.selectPostList(any())).thenReturn(posts);

        mockMvc.perform(get("/post-center/bytag/{tagId}", 1L))
                .andExpect(status().isOk());

        verify(postService, times(1)).selectPostList(any());
    }

    /**
     * 测试删除图片
     */
    @Test
    void testDeleteImage() throws Exception {
        mockMvc.perform(delete("/post-center/upload")
                .param("filename", "test.jpg"))
                .andExpect(status().isOk());
    }

    /**
     * 测试创建现有帖子的支付记录
     */
    @Test
    void testCreatePaymentForExistingPost() throws Exception {
        Post post = new Post();
        post.setPostId(1L);
        post.setAuthorId(1L);

        when(postService.selectPostById(1L)).thenReturn(post);
        when(postPaymentMapper.insertPostPayment(any())).thenReturn(1);

        String paymentData = "{\"postId\":1,\"planId\":1,\"amount\":50.0}";

        mockMvc.perform(post("/post-center/payment")
                .contentType(MediaType.APPLICATION_JSON)
                .content(paymentData))
                .andExpect(status().isOk());

        verify(postService, times(1)).selectPostById(1L);
        verify(postPaymentMapper, times(1)).insertPostPayment(any());
    }

    /**
     * 测试确认现有帖子的支付
     */
    @Test
    void testConfirmPaymentForExistingPost() throws Exception {
        PostPayment payment = new PostPayment();
        payment.setPaymentId(1L);
        payment.setPostId(1L); // 现有帖子
        payment.setUserId(1L);
        payment.setPlanId(1L);
        payment.setPaymentStatus("pending");

        when(postPaymentMapper.selectPostPaymentById(1L)).thenReturn(payment);
        when(postPaymentMapper.updatePostPayment(any())).thenReturn(1);
        when(postService.updatePost(any())).thenReturn(1);

        mockMvc.perform(post("/post-center/payment/confirm/{paymentId}", 1L))
                .andExpect(status().isOk());

        verify(postPaymentMapper, times(1)).selectPostPaymentById(1L);
        verify(postPaymentMapper, times(1)).updatePostPayment(any());
        verify(postService, times(1)).updatePost(any());
    }

    /**
     * 测试支付记录不存在的情况
     */
    @Test
    void testConfirmNonExistentPayment() throws Exception {
        when(postPaymentMapper.selectPostPaymentById(999L)).thenReturn(null);

        mockMvc.perform(post("/post-center/payment/confirm/{paymentId}", 999L))
                .andExpect(status().isOk());

        verify(postPaymentMapper, times(1)).selectPostPaymentById(999L);
        verify(postPaymentMapper, never()).updatePostPayment(any());
    }

    /**
     * 测试支付状态异常的情况
     */
    @Test
    void testConfirmPaymentWithInvalidStatus() throws Exception {
        PostPayment payment = new PostPayment();
        payment.setPaymentId(1L);
        payment.setPostId(1L);
        payment.setUserId(1L);
        payment.setPlanId(1L);
        payment.setPaymentStatus("paid"); // 已支付状态

        when(postPaymentMapper.selectPostPaymentById(1L)).thenReturn(payment);

        mockMvc.perform(post("/post-center/payment/confirm/{paymentId}", 1L))
                .andExpect(status().isOk());

        verify(postPaymentMapper, times(1)).selectPostPaymentById(1L);
        verify(postPaymentMapper, never()).updatePostPayment(any());
    }

    /**
     * 测试取消支付
     */
    @Test
    void testCancelPayment() throws Exception {
        PostPayment payment = new PostPayment();
        payment.setPaymentId(1L);
        payment.setPostId(123L); // 使用有效的帖子ID
        payment.setUserId(1L);
        payment.setPlanId(1L);
        payment.setPaymentStatus("pending");

        when(postPaymentMapper.selectPostPaymentById(1L)).thenReturn(payment);
        when(postPaymentMapper.updatePostPayment(any())).thenReturn(1);

        mockMvc.perform(post("/post-center/payment/cancel/{paymentId}", 1L))
                .andExpect(status().isOk());

        verify(postPaymentMapper, times(1)).selectPostPaymentById(1L);
        verify(postPaymentMapper, times(1)).updatePostPayment(any());
    }

    /**
     * 测试用户权限检查（只能操作自己的支付记录）
     */
    @Test
    void testPaymentUserPermission() throws Exception {
        PostPayment payment = new PostPayment();
        payment.setPaymentId(1L);
        payment.setPostId(1L);
        payment.setUserId(2L); // 不同的用户ID
        payment.setPlanId(1L);
        payment.setPaymentStatus("pending");

        when(postPaymentMapper.selectPostPaymentById(1L)).thenReturn(payment);

        mockMvc.perform(post("/post-center/payment/confirm/{paymentId}", 1L))
                .andExpect(status().isOk());

        verify(postPaymentMapper, times(1)).selectPostPaymentById(1L);
        verify(postPaymentMapper, never()).updatePostPayment(any());
    }

    /**
     * 测试新推广支付流程：发布帖子返回postId
     */
    @Test
    void testPublishPostWithReturnPostId() throws Exception {
        when(postService.insertPost(any())).thenReturn(1);

        String postData = "{\"title\":\"新帖子\",\"content\":\"帖子内容\",\"summary\":\"帖子摘要\"}";

        mockMvc.perform(post("/post-center/publish")
                .contentType(MediaType.APPLICATION_JSON)
                .content(postData))
                .andExpect(status().isOk());

        verify(postService, times(1)).insertPost(any(Post.class));
    }

    /**
     * 测试创建支付记录时检查帖子存在性
     */
    @Test
    void testCreatePaymentWithValidPost() throws Exception {
        Post post = new Post();
        post.setPostId(123L);
        post.setAuthorId(1L);

        when(postService.selectPostById(123L)).thenReturn(post);
        when(postPaymentMapper.selectLatestPaymentByPostId(123L)).thenReturn(null);
        when(postPaymentMapper.insertPostPayment(any())).thenReturn(1);

        String paymentData = "{\"postId\":123,\"planId\":1,\"amount\":50.0}";

        mockMvc.perform(post("/post-center/payment")
                .contentType(MediaType.APPLICATION_JSON)
                .content(paymentData))
                .andExpect(status().isOk());

        verify(postService, times(1)).selectPostById(123L);
        verify(postPaymentMapper, times(1)).insertPostPayment(any());
    }

    /**
     * 测试重复购买推广的检查
     */
    @Test
    void testCreatePaymentWithExistingPromotion() throws Exception {
        Post post = new Post();
        post.setPostId(123L);
        post.setAuthorId(1L);

        PostPayment existingPayment = new PostPayment();
        existingPayment.setPaymentStatus("paid");

        when(postService.selectPostById(123L)).thenReturn(post);
        when(postPaymentMapper.selectLatestPaymentByPostId(123L)).thenReturn(existingPayment);

        String paymentData = "{\"postId\":123,\"planId\":1,\"amount\":50.0}";

        mockMvc.perform(post("/post-center/payment")
                .contentType(MediaType.APPLICATION_JSON)
                .content(paymentData))
                .andExpect(status().isOk());

        verify(postService, times(1)).selectPostById(123L);
        verify(postPaymentMapper, never()).insertPostPayment(any());
    }

    /**
     * 测试支付确认时更新帖子推广信息
     */
    @Test
    void testConfirmPaymentUpdatePost() throws Exception {
        PostPayment payment = new PostPayment();
        payment.setPaymentId(1L);
        payment.setPostId(123L); // 有效的帖子ID
        payment.setUserId(1L);
        payment.setPlanId(1L);
        payment.setPaymentStatus("pending");

        when(postPaymentMapper.selectPostPaymentById(1L)).thenReturn(payment);
        when(postPaymentMapper.updatePostPayment(any())).thenReturn(1);
        when(postService.updatePost(any())).thenReturn(1);

        mockMvc.perform(post("/post-center/payment/confirm/{paymentId}", 1L))
                .andExpect(status().isOk());

        verify(postPaymentMapper, times(1)).selectPostPaymentById(1L);
        verify(postPaymentMapper, times(1)).updatePostPayment(any());
        verify(postService, times(1)).updatePost(any());
    }

    /**
     * 测试支付确认失败（无效的帖子ID）
     */
    @Test
    void testConfirmPaymentWithInvalidPostId() throws Exception {
        PostPayment payment = new PostPayment();
        payment.setPaymentId(1L);
        payment.setPostId(0L); // 无效的帖子ID
        payment.setUserId(1L);
        payment.setPlanId(1L);
        payment.setPaymentStatus("pending");

        when(postPaymentMapper.selectPostPaymentById(1L)).thenReturn(payment);
        when(postPaymentMapper.updatePostPayment(any())).thenReturn(1);

        mockMvc.perform(post("/post-center/payment/confirm/{paymentId}", 1L))
                .andExpect(status().isOk());

        verify(postPaymentMapper, times(1)).selectPostPaymentById(1L);
        verify(postPaymentMapper, times(1)).updatePostPayment(any());
        verify(postService, never()).updatePost(any());
    }

    /**
     * 测试创建支付时帖子不存在
     */
    @Test
    void testCreatePaymentWithNonExistentPost() throws Exception {
        when(postService.selectPostById(999L)).thenReturn(null);

        String paymentData = "{\"postId\":999,\"planId\":1,\"amount\":50.0}";

        mockMvc.perform(post("/post-center/payment")
                .contentType(MediaType.APPLICATION_JSON)
                .content(paymentData))
                .andExpect(status().isOk());

        verify(postService, times(1)).selectPostById(999L);
        verify(postPaymentMapper, never()).insertPostPayment(any());
    }

    /**
     * 测试创建支付时帖子不属于当前用户
     */
    @Test
    void testCreatePaymentWithOtherUserPost() throws Exception {
        Post post = new Post();
        post.setPostId(123L);
        post.setAuthorId(2L); // 不同的用户ID

        when(postService.selectPostById(123L)).thenReturn(post);

        String paymentData = "{\"postId\":123,\"planId\":1,\"amount\":50.0}";

        mockMvc.perform(post("/post-center/payment")
                .contentType(MediaType.APPLICATION_JSON)
                .content(paymentData))
                .andExpect(status().isOk());

        verify(postService, times(1)).selectPostById(123L);
        verify(postPaymentMapper, never()).insertPostPayment(any());
    }

    /**
     * 测试删除草稿帖子（状态为0）
     */
    @Test
    void testDeleteDraftPost() throws Exception {
        Post draftPost = new Post();
        draftPost.setPostId(123L);
        draftPost.setAuthorId(1L);
        draftPost.setStatus("0"); // 草稿状态

        when(postService.selectPostById(123L)).thenReturn(draftPost);
        when(postPaymentMapper.deletePostPaymentByPostId(123L)).thenReturn(1);
        when(postService.deletePostById(123L)).thenReturn(1);

        mockMvc.perform(delete("/post-center/delete/{postId}", 123L))
                .andExpect(status().isOk());

        verify(postService, times(1)).selectPostById(123L);
        verify(postPaymentMapper, times(1)).deletePostPaymentByPostId(123L);
        verify(postService, times(1)).deletePostById(123L);
        // 草稿帖子不需要删除评论和收藏
        verify(postCommentService, never()).deletePostCommentByPostId(any());
        verify(postFavoriteService, never()).deletePostFavoriteByPostId(any());
    }

    /**
     * 测试删除已发布帖子（状态为1）
     */
    @Test
    void testDeletePublishedPost() throws Exception {
        Post publishedPost = new Post();
        publishedPost.setPostId(123L);
        publishedPost.setAuthorId(1L);
        publishedPost.setStatus("1"); // 已发布状态

        when(postService.selectPostById(123L)).thenReturn(publishedPost);
        when(postCommentService.deletePostCommentByPostId(123L)).thenReturn(1);
        when(postFavoriteService.deletePostFavoriteByPostId(123L)).thenReturn(1);
        when(postPaymentMapper.deletePostPaymentByPostId(123L)).thenReturn(1);
        when(postService.deletePostById(123L)).thenReturn(1);

        mockMvc.perform(delete("/post-center/delete/{postId}", 123L))
                .andExpect(status().isOk());

        verify(postService, times(1)).selectPostById(123L);
        verify(postCommentService, times(1)).deletePostCommentByPostId(123L);
        verify(postFavoriteService, times(1)).deletePostFavoriteByPostId(123L);
        verify(postPaymentMapper, times(1)).deletePostPaymentByPostId(123L);
        verify(postService, times(1)).deletePostById(123L);
    }

    /**
     * 测试重复创建支付记录的逻辑
     */
    @Test
    void testMultiplePaymentAttempts() throws Exception {
        Post post = new Post();
        post.setPostId(123L);
        post.setAuthorId(1L);

        // 第一次创建支付记录（没有现有支付记录）
        when(postService.selectPostById(123L)).thenReturn(post);
        when(postPaymentMapper.selectLatestPaymentByPostId(123L)).thenReturn(null);
        when(postPaymentMapper.insertPostPayment(any())).thenReturn(1);

        String paymentData = "{\"postId\":123,\"planId\":1,\"amount\":50.0}";

        mockMvc.perform(post("/post-center/payment")
                .contentType(MediaType.APPLICATION_JSON)
                .content(paymentData))
                .andExpect(status().isOk());

        verify(postPaymentMapper, times(1)).insertPostPayment(any());

        // 第二次创建支付记录（有pending状态的支付记录，应该可以创建新的）
        PostPayment pendingPayment = new PostPayment();
        pendingPayment.setPaymentStatus("pending");

        when(postPaymentMapper.selectLatestPaymentByPostId(123L)).thenReturn(pendingPayment);

        mockMvc.perform(post("/post-center/payment")
                .contentType(MediaType.APPLICATION_JSON)
                .content(paymentData))
                .andExpect(status().isOk());

        verify(postPaymentMapper, times(2)).insertPostPayment(any());
    }

    /**
     * 测试支付失败后重新创建支付记录
     */
    @Test
    void testRetryPaymentAfterFailure() throws Exception {
        Post post = new Post();
        post.setPostId(123L);
        post.setAuthorId(1L);

        PostPayment failedPayment = new PostPayment();
        failedPayment.setPaymentStatus("failed");

        when(postService.selectPostById(123L)).thenReturn(post);
        when(postPaymentMapper.selectLatestPaymentByPostId(123L)).thenReturn(failedPayment);
        when(postPaymentMapper.insertPostPayment(any())).thenReturn(1);

        String paymentData = "{\"postId\":123,\"planId\":1,\"amount\":50.0}";

        mockMvc.perform(post("/post-center/payment")
                .contentType(MediaType.APPLICATION_JSON)
                .content(paymentData))
                .andExpect(status().isOk());

        verify(postService, times(1)).selectPostById(123L);
        verify(postPaymentMapper, times(1)).selectLatestPaymentByPostId(123L);
        verify(postPaymentMapper, times(1)).insertPostPayment(any());
    }
} 