package com.ruoyi.web.controller.post;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.ruoyi.common.core.domain.entity.SysUser;
import com.ruoyi.common.core.domain.model.LoginUser;
import com.ruoyi.web.controller.post.controller.PostCommentController;
import com.ruoyi.web.controller.post.domain.PostComment;
import com.ruoyi.web.controller.post.service.IPostCommentService;
import com.ruoyi.web.controller.post.service.IPostService;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.http.MediaType;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

import static org.mockito.ArgumentMatchers.*;
import static org.mockito.Mockito.*;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.*;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.*;
import org.springframework.boot.test.context.SpringBootTestContextBootstrapper;
import org.springframework.test.context.BootstrapWith;
/**
 * PostCommentController 单元测试
 * 
 * 测试覆盖所有评论管理相关的API接口
 * 包括：评论列表、详情、增删改查、点赞、导出等功能
 */
@ExtendWith(MockitoExtension.class)
class PostCommentControllerTest {

    @Mock
    private IPostCommentService postCommentService;

    @Mock
    private IPostService postService;

    @InjectMocks
    private PostCommentController postCommentController;

    private MockMvc mockMvc;
    private ObjectMapper objectMapper;

    @BeforeEach
    void setUp() {
        // 创建模拟的安全上下文
        SysUser sysUser = new SysUser();
        sysUser.setUserId(1L);
        sysUser.setUserName("testUser");
        
        LoginUser loginUser = new LoginUser(1L, 1L, sysUser, new HashSet<>());
        
        UsernamePasswordAuthenticationToken authToken = 
            new UsernamePasswordAuthenticationToken(loginUser, null, loginUser.getAuthorities());
        SecurityContextHolder.getContext().setAuthentication(authToken);
        
        mockMvc = MockMvcBuilders.standaloneSetup(postCommentController).build();
        objectMapper = new ObjectMapper();
    }

    /**
     * 测试获取评论列表
     */
    @Test
    void testGetCommentList() throws Exception {
        List<PostComment> comments = new ArrayList<>();
        PostComment comment = new PostComment();
        comment.setCommentId(1L);
        comment.setContent("测试评论");
        comments.add(comment);

        when(postCommentService.selectPostCommentList(any(PostComment.class))).thenReturn(comments);

        mockMvc.perform(get("/post/comment/list"))
                .andExpect(status().isOk());

        verify(postCommentService, times(1)).selectPostCommentList(any(PostComment.class));
    }

    /**
     * 测试获取评论详情
     */
    @Test
    void testGetCommentInfo() throws Exception {
        PostComment comment = new PostComment();
        comment.setCommentId(1L);
        comment.setContent("测试评论");

        when(postCommentService.selectPostCommentById(1L)).thenReturn(comment);

        mockMvc.perform(get("/post/comment/{commentId}", 1L))
                .andExpect(status().isOk());

        verify(postCommentService, times(1)).selectPostCommentById(1L);
    }

    /**
     * 测试添加评论
     */
    @Test
    void testAddComment() throws Exception {
        PostComment comment = new PostComment();
        comment.setPostId(1L);
        comment.setContent("新评论");

        when(postCommentService.insertPostComment(any(PostComment.class))).thenReturn(1);
        when(postService.updatePostComments(eq(1L), eq(1))).thenReturn(1);

        mockMvc.perform(post("/post/comment")
                .contentType(MediaType.APPLICATION_JSON)
                .content(objectMapper.writeValueAsString(comment)))
                .andExpect(status().isOk());

        verify(postCommentService, times(1)).insertPostComment(any(PostComment.class));
        verify(postService, times(1)).updatePostComments(eq(1L), eq(1));
    }

    /**
     * 测试编辑评论
     */
    @Test
    void testEditComment() throws Exception {
        PostComment comment = new PostComment();
        comment.setCommentId(1L);
        comment.setContent("编辑后的评论");

        when(postCommentService.updatePostComment(any(PostComment.class))).thenReturn(1);

        mockMvc.perform(put("/post/comment")
                .contentType(MediaType.APPLICATION_JSON)
                .content(objectMapper.writeValueAsString(comment)))
                .andExpect(status().isOk());

        verify(postCommentService, times(1)).updatePostComment(any(PostComment.class));
    }

    /**
     * 测试删除评论
     */
    @Test
    void testDeleteComment() throws Exception {
        PostComment comment = new PostComment();
        comment.setCommentId(1L);
        comment.setPostId(1L);
        comment.setParentId(0L);

        when(postCommentService.selectPostCommentById(1L)).thenReturn(comment);
        when(postCommentService.selectCommentsByParentId(1L)).thenReturn(new ArrayList<>());
        when(postCommentService.deletePostCommentByIds(any())).thenReturn(1);
        when(postService.updatePostComments(eq(1L), eq(-1))).thenReturn(1);

        mockMvc.perform(delete("/post/comment/{commentIds}", "1"))
                .andExpect(status().isOk());

        verify(postCommentService, times(1)).selectPostCommentById(1L);
        verify(postCommentService, times(1)).deletePostCommentByIds(any());
        verify(postService, times(1)).updatePostComments(eq(1L), eq(-1));
    }

    /**
     * 测试删除包含回复的顶级评论
     */
    @Test
    void testDeleteTopCommentWithReplies() throws Exception {
        PostComment comment = new PostComment();
        comment.setCommentId(1L);
        comment.setPostId(1L);
        comment.setParentId(0L);

        List<PostComment> replies = new ArrayList<>();
        PostComment reply1 = new PostComment();
        reply1.setCommentId(2L);
        replies.add(reply1);
        PostComment reply2 = new PostComment();
        reply2.setCommentId(3L);
        replies.add(reply2);

        when(postCommentService.selectPostCommentById(1L)).thenReturn(comment);
        when(postCommentService.selectCommentsByParentId(1L)).thenReturn(replies);
        when(postCommentService.deletePostCommentByIds(any())).thenReturn(1);
        when(postService.updatePostComments(eq(1L), eq(-3))).thenReturn(1); // 1个评论 + 2个回复

        mockMvc.perform(delete("/post/comment/{commentIds}", "1"))
                .andExpect(status().isOk());

        verify(postCommentService, times(1)).selectPostCommentById(1L);
        verify(postCommentService, times(1)).selectCommentsByParentId(1L);
        verify(postCommentService, times(1)).deletePostCommentByIds(any());
        verify(postService, times(1)).updatePostComments(eq(1L), eq(-3));
    }

    /**
     * 测试根据帖子ID获取评论列表
     */
    @Test
    void testListByPostId() throws Exception {
        List<PostComment> comments = new ArrayList<>();
        when(postCommentService.selectCommentsByPostId(eq(1L), eq(0))).thenReturn(comments);

        mockMvc.perform(get("/post/comment/list/{postId}", 1L))
                .andExpect(status().isOk());

        verify(postCommentService, times(1)).selectCommentsByPostId(eq(1L), eq(0));
    }

    /**
     * 测试点赞评论
     */
    @Test
    void testLikeComment() throws Exception {
        when(postCommentService.updateCommentLikes(eq(1L), eq(1))).thenReturn(1);

        mockMvc.perform(post("/post/comment/like/{commentId}", 1L))
                .andExpect(status().isOk());

        verify(postCommentService, times(1)).updateCommentLikes(eq(1L), eq(1));
    }

    /**
     * 测试导出评论
     */
    @Test
    void testExportComments() throws Exception {
        List<PostComment> comments = new ArrayList<>();
        when(postCommentService.selectPostCommentList(any(PostComment.class))).thenReturn(comments);

        mockMvc.perform(post("/post/comment/export"))
                .andExpect(status().isOk());

        verify(postCommentService, times(1)).selectPostCommentList(any(PostComment.class));
    }

    /**
     * 测试添加回复评论（非顶级评论）
     */
    @Test
    void testAddReplyComment() throws Exception {
        PostComment replyComment = new PostComment();
        replyComment.setPostId(1L);
        replyComment.setContent("回复评论");
        replyComment.setParentId(1L); // 回复给评论ID为1的评论

        when(postCommentService.insertPostComment(any(PostComment.class))).thenReturn(1);
        when(postService.updatePostComments(eq(1L), eq(1))).thenReturn(1);

        mockMvc.perform(post("/post/comment")
                .contentType(MediaType.APPLICATION_JSON)
                .content(objectMapper.writeValueAsString(replyComment)))
                .andExpect(status().isOk());

        verify(postCommentService, times(1)).insertPostComment(any(PostComment.class));
        verify(postService, times(1)).updatePostComments(eq(1L), eq(1));
    }

    /**
     * 测试删除不存在的评论
     */
    @Test
    void testDeleteNonExistentComment() throws Exception {
        when(postCommentService.selectPostCommentById(999L)).thenReturn(null);
        when(postCommentService.deletePostCommentByIds(any())).thenReturn(0);

        mockMvc.perform(delete("/post/comment/{commentIds}", "999"))
                .andExpect(status().isOk());

        verify(postCommentService, times(1)).selectPostCommentById(999L);
        verify(postCommentService, times(1)).deletePostCommentByIds(any());
        verify(postService, never()).updatePostComments(any(), anyInt());
    }

    /**
     * 测试批量删除评论
     */
    @Test
    void testBatchDeleteComments() throws Exception {
        PostComment comment1 = new PostComment();
        comment1.setCommentId(1L);
        comment1.setPostId(1L);
        comment1.setParentId(0L);

        PostComment comment2 = new PostComment();
        comment2.setCommentId(2L);
        comment2.setPostId(1L);
        comment2.setParentId(0L);

        when(postCommentService.selectPostCommentById(1L)).thenReturn(comment1);
        when(postCommentService.selectPostCommentById(2L)).thenReturn(comment2);
        when(postCommentService.selectCommentsByParentId(1L)).thenReturn(new ArrayList<>());
        when(postCommentService.selectCommentsByParentId(2L)).thenReturn(new ArrayList<>());
        when(postCommentService.deletePostCommentByIds(any())).thenReturn(2);
        when(postService.updatePostComments(eq(1L), eq(-1))).thenReturn(1);

        mockMvc.perform(delete("/post/comment/{commentIds}", "1,2"))
                .andExpect(status().isOk());

        verify(postCommentService, times(1)).selectPostCommentById(1L);
        verify(postCommentService, times(1)).selectPostCommentById(2L);
        verify(postCommentService, times(1)).deletePostCommentByIds(any());
        verify(postService, times(2)).updatePostComments(eq(1L), eq(-1));
    }

    /**
     * 测试添加空内容评论（应该返回验证错误）
     */
    @Test
    void testAddEmptyContentComment() throws Exception {
        PostComment comment = new PostComment();
        comment.setPostId(1L);
        comment.setContent(""); // 空内容

        mockMvc.perform(post("/post/comment")
                .contentType(MediaType.APPLICATION_JSON)
                .content(objectMapper.writeValueAsString(comment)))
                .andExpect(status().isBadRequest());

        verify(postCommentService, never()).insertPostComment(any(PostComment.class));
    }

    /**
     * 测试更新评论状态
     */
    @Test
    void testUpdateCommentStatus() throws Exception {
        PostComment comment = new PostComment();
        comment.setCommentId(1L);
        comment.setStatus("0");
        comment.setContent("有效内容");

        when(postCommentService.updatePostComment(any(PostComment.class))).thenReturn(1);

        mockMvc.perform(put("/post/comment")
                .contentType(MediaType.APPLICATION_JSON)
                .content(objectMapper.writeValueAsString(comment)))
                .andExpect(status().isOk());

        verify(postCommentService, times(1)).updatePostComment(any(PostComment.class));
    }

    /**
     * 测试获取不存在评论的详情
     */
    @Test
    void testGetNonExistentCommentInfo() throws Exception {
        when(postCommentService.selectPostCommentById(999L)).thenReturn(null);

        mockMvc.perform(get("/post/comment/{commentId}", 999L))
                .andExpect(status().isOk());

        verify(postCommentService, times(1)).selectPostCommentById(999L);
    }

    /**
     * 测试编辑不存在的评论
     */
    @Test
    void testEditNonExistentComment() throws Exception {
        PostComment comment = new PostComment();
        comment.setCommentId(999L);
        comment.setContent("不存在的评论");

        when(postCommentService.updatePostComment(any(PostComment.class))).thenReturn(0);

        mockMvc.perform(put("/post/comment")
                .contentType(MediaType.APPLICATION_JSON)
                .content(objectMapper.writeValueAsString(comment)))
                .andExpect(status().isOk());

        verify(postCommentService, times(1)).updatePostComment(any(PostComment.class));
    }
} 