package com.ruoyi.web.controller.post;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.ruoyi.common.core.domain.entity.SysUser;
import com.ruoyi.common.core.domain.model.LoginUser;
import com.ruoyi.web.controller.post.controller.PostTagController;
import com.ruoyi.web.controller.post.domain.PostTag;
import com.ruoyi.web.controller.post.service.IPostTagService;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.http.MediaType;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

import static org.mockito.ArgumentMatchers.*;
import static org.mockito.Mockito.*;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.*;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.*;
import static org.mockito.Mockito.lenient;
import org.springframework.boot.test.context.SpringBootTestContextBootstrapper;
import org.springframework.test.context.BootstrapWith;
/**
 * PostTagController 单元测试
 * 
 * 测试覆盖所有标签管理相关的API接口
 * 包括：标签列表、详情、增删改查、导出等功能
 */
@ExtendWith(MockitoExtension.class)
class PostTagControllerTest {

    @Mock
    private IPostTagService postTagService;

    @InjectMocks
    private PostTagController postTagController;

    private MockMvc mockMvc;
    private ObjectMapper objectMapper;

    @BeforeEach
    void setUp() {
        // 创建模拟的安全上下文
        SysUser sysUser = new SysUser();
        sysUser.setUserId(1L);
        sysUser.setUserName("testUser");
        
        LoginUser loginUser = new LoginUser(1L, 1L, sysUser, new HashSet<>());
        
        UsernamePasswordAuthenticationToken authToken = 
            new UsernamePasswordAuthenticationToken(loginUser, null, loginUser.getAuthorities());
        SecurityContextHolder.getContext().setAuthentication(authToken);
        
        mockMvc = MockMvcBuilders.standaloneSetup(postTagController).build();
        objectMapper = new ObjectMapper();
    }

    /**
     * 测试获取标签列表
     */
    @Test
    void testGetTagList() throws Exception {
        List<PostTag> tags = new ArrayList<>();
        PostTag tag = new PostTag();
        tag.setTagId(1L);
        tag.setTagName("测试标签");
        tags.add(tag);

        when(postTagService.selectPostTagList(any(PostTag.class))).thenReturn(tags);

        mockMvc.perform(get("/post/tag/list"))
                .andExpect(status().isOk());

        verify(postTagService, times(1)).selectPostTagList(any(PostTag.class));
    }

    /**
     * 测试获取标签详情
     */
    @Test
    void testGetTagInfo() throws Exception {
        PostTag tag = new PostTag();
        tag.setTagId(1L);
        tag.setTagName("测试标签");

        when(postTagService.selectPostTagById(1L)).thenReturn(tag);

        mockMvc.perform(get("/post/tag/{tagId}", 1L))
                .andExpect(status().isOk());

        verify(postTagService, times(1)).selectPostTagById(1L);
    }

    /**
     * 测试添加标签
     */
    @Test
    void testAddTag() throws Exception {
        PostTag tag = new PostTag();
        tag.setTagName("新标签");

        when(postTagService.checkTagNameUnique(any())).thenReturn(true);
        when(postTagService.insertPostTag(any(PostTag.class))).thenReturn(1);

        mockMvc.perform(post("/post/tag")
                .contentType(MediaType.APPLICATION_JSON)
                .content(objectMapper.writeValueAsString(tag)))
                .andExpect(status().isOk());

        verify(postTagService, times(1)).checkTagNameUnique(any());
        verify(postTagService, times(1)).insertPostTag(any(PostTag.class));
    }

    /**
     * 测试编辑标签
     */
    @Test
    void testEditTag() throws Exception {
        PostTag tag = new PostTag();
        tag.setTagId(1L);
        tag.setTagName("编辑后的标签");

        when(postTagService.checkTagNameUnique(any())).thenReturn(true);
        when(postTagService.updatePostTag(any(PostTag.class))).thenReturn(1);

        mockMvc.perform(put("/post/tag")
                .contentType(MediaType.APPLICATION_JSON)
                .content(objectMapper.writeValueAsString(tag)))
                .andExpect(status().isOk());

        verify(postTagService, times(1)).checkTagNameUnique(any());
        verify(postTagService, times(1)).updatePostTag(any(PostTag.class));
    }

    /**
     * 测试删除标签
     */
    @Test
    void testDeleteTag() throws Exception {
        when(postTagService.deletePostTagByIds(any())).thenReturn(1);

        mockMvc.perform(delete("/post/tag/{tagIds}", "1"))
                .andExpect(status().isOk());

        verify(postTagService, times(1)).deletePostTagByIds(any());
    }

    /**
     * 测试批量删除标签
     */
    @Test
    void testBatchDeleteTags() throws Exception {
        when(postTagService.deletePostTagByIds(any())).thenReturn(2);

        mockMvc.perform(delete("/post/tag/{tagIds}", "1,2"))
                .andExpect(status().isOk());

        verify(postTagService, times(1)).deletePostTagByIds(any());
    }

    /**
     * 测试导出标签
     */
    @Test
    void testExportTags() throws Exception {
        List<PostTag> tags = new ArrayList<>();
        when(postTagService.selectPostTagList(any(PostTag.class))).thenReturn(tags);

        mockMvc.perform(post("/post/tag/export"))
                .andExpect(status().isOk());

        verify(postTagService, times(1)).selectPostTagList(any(PostTag.class));
    }

    /**
     * 测试根据帖子ID获取标签列表
     */
    @Test
    void testListByPostId() throws Exception {
        List<PostTag> tags = new ArrayList<>();
        when(postTagService.selectPostTagsByPostId(1L)).thenReturn(tags);

        mockMvc.perform(get("/post/tag/list/{postId}", 1L))
                .andExpect(status().isOk());

        verify(postTagService, times(1)).selectPostTagsByPostId(1L);
    }

    /**
     * 测试获取可用标签选项
     */
    @Test
    void testOptionSelect() throws Exception {
        List<PostTag> tags = new ArrayList<>();
        PostTag tag = new PostTag();
        tag.setTagId(1L);
        tag.setTagName("可用标签");
        tag.setStatus("0");
        tags.add(tag);

        when(postTagService.selectPostTagList(any(PostTag.class))).thenReturn(tags);

        mockMvc.perform(get("/post/tag/optionselect"))
                .andExpect(status().isOk());

        verify(postTagService, times(1)).selectPostTagList(any(PostTag.class));
    }

    /**
     * 测试获取不存在的标签详情
     */
    @Test
    void testGetNonExistentTagInfo() throws Exception {
        when(postTagService.selectPostTagById(999L)).thenReturn(null);

        mockMvc.perform(get("/post/tag/{tagId}", 999L))
                .andExpect(status().isOk());

        verify(postTagService, times(1)).selectPostTagById(999L);
    }

    /**
     * 测试编辑不存在的标签
     */
    @Test
    void testEditNonExistentTag() throws Exception {
        PostTag tag = new PostTag();
        tag.setTagId(999L);
        tag.setTagName("不存在的标签");

        when(postTagService.checkTagNameUnique(any())).thenReturn(true);
        when(postTagService.updatePostTag(any(PostTag.class))).thenReturn(0);

        mockMvc.perform(put("/post/tag")
                .contentType(MediaType.APPLICATION_JSON)
                .content(objectMapper.writeValueAsString(tag)))
                .andExpect(status().isOk());

        verify(postTagService, times(1)).updatePostTag(any(PostTag.class));
    }

    /**
     * 测试删除不存在的标签
     */
    @Test
    void testDeleteNonExistentTag() throws Exception {
        when(postTagService.deletePostTagByIds(any())).thenReturn(0);

        mockMvc.perform(delete("/post/tag/{tagIds}", "999"))
                .andExpect(status().isOk());

        verify(postTagService, times(1)).deletePostTagByIds(any());
    }

    /**
     * 测试添加重复标签名
     */
    @Test
    void testAddDuplicateTagName() throws Exception {
        PostTag tag = new PostTag();
        tag.setTagName("重复标签");

        when(postTagService.checkTagNameUnique(any())).thenReturn(false);

        mockMvc.perform(post("/post/tag")
                .contentType(MediaType.APPLICATION_JSON)
                .content(objectMapper.writeValueAsString(tag)))
                .andExpect(status().isOk());

        verify(postTagService, times(1)).checkTagNameUnique(any());
        verify(postTagService, never()).insertPostTag(any());
    }

    /**
     * 测试编辑为重复标签名
     */
    @Test
    void testEditToDuplicateTagName() throws Exception {
        PostTag tag = new PostTag();
        tag.setTagId(1L);
        tag.setTagName("重复标签");

        when(postTagService.checkTagNameUnique(any())).thenReturn(false);

        mockMvc.perform(put("/post/tag")
                .contentType(MediaType.APPLICATION_JSON)
                .content(objectMapper.writeValueAsString(tag)))
                .andExpect(status().isOk());

        verify(postTagService, times(1)).checkTagNameUnique(any());
        verify(postTagService, never()).updatePostTag(any());
    }

    /**
     * 测试添加标签失败
     */
    @Test
    void testAddTagFailed() throws Exception {
        PostTag tag = new PostTag();
        tag.setTagName("新标签");

        when(postTagService.checkTagNameUnique(any())).thenReturn(true);
        when(postTagService.insertPostTag(any(PostTag.class))).thenReturn(0);

        mockMvc.perform(post("/post/tag")
                .contentType(MediaType.APPLICATION_JSON)
                .content(objectMapper.writeValueAsString(tag)))
                .andExpect(status().isOk());

        verify(postTagService, times(1)).checkTagNameUnique(any());
        verify(postTagService, times(1)).insertPostTag(any(PostTag.class));
    }

    /**
     * 测试编辑标签失败
     */
    @Test
    void testEditTagFailed() throws Exception {
        PostTag tag = new PostTag();
        tag.setTagId(1L);
        tag.setTagName("编辑后的标签");

        when(postTagService.checkTagNameUnique(any())).thenReturn(true);
        when(postTagService.updatePostTag(any(PostTag.class))).thenReturn(0);

        mockMvc.perform(put("/post/tag")
                .contentType(MediaType.APPLICATION_JSON)
                .content(objectMapper.writeValueAsString(tag)))
                .andExpect(status().isOk());

        verify(postTagService, times(1)).checkTagNameUnique(any());
        verify(postTagService, times(1)).updatePostTag(any(PostTag.class));
    }

    /**
     * 测试删除标签失败
     */
    @Test
    void testDeleteTagFailed() throws Exception {
        when(postTagService.deletePostTagByIds(any())).thenReturn(0);

        mockMvc.perform(delete("/post/tag/{tagIds}", "1"))
                .andExpect(status().isOk());

        verify(postTagService, times(1)).deletePostTagByIds(any());
    }

    /**
     * 测试获取空的标签列表
     */
    @Test
    void testGetEmptyTagList() throws Exception {
        List<PostTag> tags = new ArrayList<>();
        when(postTagService.selectPostTagList(any(PostTag.class))).thenReturn(tags);

        mockMvc.perform(get("/post/tag/list"))
                .andExpect(status().isOk());

        verify(postTagService, times(1)).selectPostTagList(any(PostTag.class));
    }

    /**
     * 测试根据不存在的帖子ID获取标签列表
     */
    @Test
    void testListByNonExistentPostId() throws Exception {
        List<PostTag> tags = new ArrayList<>();
        when(postTagService.selectPostTagsByPostId(999L)).thenReturn(tags);

        mockMvc.perform(get("/post/tag/list/{postId}", 999L))
                .andExpect(status().isOk());

        verify(postTagService, times(1)).selectPostTagsByPostId(999L);
    }
} 