
// BountySubmissionControllerTest.java
package com.ruoyi.web.controller.bounty;

import bounty.controller.BountySubmissionController;
import bounty.domain.BountySubmission;
import bounty.service.BountySubmissionService;
import bounty.service.FileStorageService;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.ruoyi.common.core.domain.entity.SysUser;
import com.ruoyi.common.core.domain.model.LoginUser;
import org.hamcrest.Matchers;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.mock.web.MockMultipartFile;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.request.MockMvcRequestBuilders;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;
import org.springframework.web.multipart.MultipartFile;

import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;

import static org.mockito.ArgumentMatchers.*;
import static org.mockito.Mockito.*;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.*;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.*;

@ExtendWith(MockitoExtension.class)
class BountySubmissionControllerTest {

    @Mock
    private BountySubmissionService bountySubmissionService;

    @Mock
    private FileStorageService fileStorageService;

    @InjectMocks
    private BountySubmissionController bountySubmissionController;

    private MockMvc mockMvc;
    private ObjectMapper objectMapper;

    @BeforeEach
    void setUp() {
        // 创建模拟的安全上下文
        SysUser sysUser = new SysUser();
        sysUser.setUserId(1L);
        sysUser.setUserName("testUser");

        LoginUser loginUser = new LoginUser(1L, 1L, sysUser, new HashSet<>());

        UsernamePasswordAuthenticationToken authToken =
                new UsernamePasswordAuthenticationToken(loginUser, null, loginUser.getAuthorities());
        SecurityContextHolder.getContext().setAuthentication(authToken);

        mockMvc = MockMvcBuilders.standaloneSetup(bountySubmissionController).build();
        objectMapper = new ObjectMapper();
    }

    @Test
    void testSaveBountySubmissionSuccess() throws Exception {
        BountySubmission submission = new BountySubmission();
        submission.setBountyId(1L);

        String submissionJson = objectMapper.writeValueAsString(submission);

        when(fileStorageService.saveFile(any(MultipartFile.class))).thenReturn("/uploads/test.txt");
        when(bountySubmissionService.saveBountySubmission(any(BountySubmission.class))).thenReturn(true);

        mockMvc.perform(MockMvcRequestBuilders.multipart("/bounty-submissions")
                        .file(new MockMultipartFile("submission", "", MediaType.APPLICATION_JSON_VALUE, submissionJson.getBytes()))
                        .file("file", "test content".getBytes())
                        .contentType(MediaType.MULTIPART_FORM_DATA))
                .andExpect(status().isOk());

        verify(fileStorageService, times(1)).saveFile(any(MultipartFile.class));
        verify(bountySubmissionService, times(1)).saveBountySubmission(any(BountySubmission.class));
    }

    @Test
    void testGetBountySubmissions() throws Exception {
        // 构造分页对象
        Page<BountySubmission> page = new Page<>(1, 10);
        page.setTotal(0);
        page.setRecords(new ArrayList<>());

        // 模拟 service 返回空分页
        when(bountySubmissionService.getBountySubmissionsByPage(any(Page.class), eq(1L))).thenReturn(page);

        // 发起 GET 请求
        mockMvc.perform(get("/bounty-submissions")
                        .param("current", "1")
                        .param("size", "10")
                        .param("bountyId", "1"))
                .andExpect(status().isOk());
    }

    @Test
    void testAdoptSubmissionSuccess() throws Exception {
        when(bountySubmissionService.adoptSubmission(1L)).thenReturn(true);

        mockMvc.perform(put("/bounty-submissions/1/adopt"))
                .andExpect(status().isOk());

        verify(bountySubmissionService, times(1)).adoptSubmission(1L);
    }

    @Test
    void testAdoptNonExistentSubmission() throws Exception {
        when(bountySubmissionService.adoptSubmission(999L)).thenReturn(false);

        mockMvc.perform(put("/bounty-submissions/999/adopt"))
                .andExpect(status().isInternalServerError());

        verify(bountySubmissionService, times(1)).adoptSubmission(999L);
    }

    @Test
    void testUploadAttachment() throws Exception {
        when(fileStorageService.saveFile(any(MultipartFile.class))).thenReturn("/uploads/test.jpg");

        mockMvc.perform(multipart("/bounty-submissions/upload")
                        .file("file", "test content".getBytes())
                        .param("filename", "test.jpg"))
                .andExpect(status().isOk());

        verify(fileStorageService, times(1)).saveFile(any(MultipartFile.class));
    }

    @Test
    void testDownloadAttachment() throws Exception {
        Path uploadDir = Paths.get("uploads/");
        if (!Files.exists(uploadDir)) {
            Files.createDirectories(uploadDir);
        }

        Path testFilePath = uploadDir.resolve("test.jpg");
        if (!Files.exists(testFilePath)) {
            Files.write(testFilePath, "test content".getBytes());
        }

       mockMvc.perform(get("/bounty-submissions/download")
                        .param("filename", "test.jpg"))
                .andExpect(status().isOk())
                .andExpect(header().string(HttpHeaders.CONTENT_DISPOSITION,
                        Matchers.matchesPattern(".*filename=.*test\\.jpg.*")));
    }
}