package com.ruoyi.web.controller.post;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.ruoyi.common.core.domain.entity.SysUser;
import com.ruoyi.common.core.domain.model.LoginUser;
import com.ruoyi.web.controller.post.controller.PostController;
import com.ruoyi.web.controller.post.domain.Post;
import com.ruoyi.web.controller.post.domain.PostComment;
import com.ruoyi.web.controller.post.domain.PostFavorite;
import com.ruoyi.web.controller.post.domain.PostReport;
import com.ruoyi.web.controller.post.domain.PostTag;
import com.ruoyi.web.controller.post.service.IPostService;
import com.ruoyi.web.controller.post.service.IPostTagService;
import com.ruoyi.web.controller.post.service.IPostCommentService;
import com.ruoyi.web.controller.post.service.IPostFavoriteService;
import com.ruoyi.web.controller.post.mapper.PostReportMapper;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.http.MediaType;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

import static org.mockito.ArgumentMatchers.*;
import static org.mockito.Mockito.*;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.*;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.*;
import org.springframework.boot.test.context.SpringBootTestContextBootstrapper;
import org.springframework.test.context.BootstrapWith;
/**
 * PostController 单元测试
 * 
 * 测试覆盖所有帖子管理相关的API接口
 * 包括：帖子列表、详情、增删改查、审核、导出等功能
 */
@ExtendWith(MockitoExtension.class)
class PostControllerTest {

    @Mock
    private IPostService postService;

    @Mock
    private IPostTagService postTagService;

    @Mock
    private IPostCommentService postCommentService;

    @Mock
    private IPostFavoriteService postFavoriteService;

    @Mock
    private PostReportMapper postReportMapper;

    @InjectMocks
    private PostController postController;

    private MockMvc mockMvc;
    private ObjectMapper objectMapper;

    @BeforeEach
    void setUp() {
        // 创建模拟的安全上下文
        SysUser sysUser = new SysUser();
        sysUser.setUserId(1L);
        sysUser.setUserName("testUser");
        
        LoginUser loginUser = new LoginUser(1L, 1L, sysUser, new HashSet<>());
        
        UsernamePasswordAuthenticationToken authToken = 
            new UsernamePasswordAuthenticationToken(loginUser, null, loginUser.getAuthorities());
        SecurityContextHolder.getContext().setAuthentication(authToken);
        
        mockMvc = MockMvcBuilders.standaloneSetup(postController).build();
        objectMapper = new ObjectMapper();
    }

    /**
     * 测试获取帖子列表
     */
    @Test
    void testGetPostList() throws Exception {
        List<Post> posts = new ArrayList<>();
        Post post = new Post();
        post.setPostId(1L);
        post.setTitle("测试帖子");
        posts.add(post);

        when(postService.selectPostList(any(Post.class))).thenReturn(posts);

        mockMvc.perform(get("/post/list"))
                .andExpect(status().isOk());

        verify(postService, times(1)).selectPostList(any(Post.class));
    }

    /**
     * 测试获取帖子详情
     */
    @Test
    void testGetPostInfo() throws Exception {
        Post post = new Post();
        post.setPostId(1L);
        post.setTitle("测试帖子");
        post.setAuthorId(1L);
        post.setTags("测试标签");

        when(postService.updatePostViews(1L)).thenReturn(1);
        when(postService.selectPostById(1L)).thenReturn(post);
        when(postTagService.selectPostTagsByPostId(1L)).thenReturn(new ArrayList<>());
        when(postCommentService.selectPostCommentList(any())).thenReturn(new ArrayList<>());
        when(postService.selectAuthorOtherPosts(eq(1L), eq(1L), eq(3))).thenReturn(new ArrayList<>());
        when(postService.selectSimilarTagsPosts(eq("测试标签"), eq(1L), eq(3))).thenReturn(new ArrayList<>());
        when(postFavoriteService.selectPostFavoriteByPostIdAndUserId(eq(1L), eq(1L))).thenReturn(null);

        mockMvc.perform(get("/post/{postId}", 1L))
                .andExpect(status().isOk());

        verify(postService, times(1)).selectPostById(1L);
        verify(postService, times(1)).updatePostViews(1L);
    }

    /**
     * 测试添加帖子
     */
    @Test
    void testAddPost() throws Exception {
        Post post = new Post();
        post.setTitle("新帖子");
        post.setContent("帖子内容");

        when(postService.insertPost(any(Post.class))).thenReturn(1);

        mockMvc.perform(post("/post")
                .contentType(MediaType.APPLICATION_JSON)
                .content(objectMapper.writeValueAsString(post)))
                .andExpect(status().isOk());

        verify(postService, times(1)).insertPost(any(Post.class));
    }

    /**
     * 测试编辑帖子
     */
    @Test
    void testEditPost() throws Exception {
        Post post = new Post();
        post.setPostId(1L);
        post.setTitle("编辑后的帖子");

        when(postService.updatePost(any(Post.class))).thenReturn(1);
        when(postTagService.deletePostTagRelation(1L)).thenReturn(1);

        mockMvc.perform(put("/post")
                .contentType(MediaType.APPLICATION_JSON)
                .content(objectMapper.writeValueAsString(post)))
                .andExpect(status().isOk());

        verify(postService, times(1)).updatePost(any(Post.class));
    }

    /**
     * 测试删除帖子
     */
    @Test
    void testDeletePost() throws Exception {
        when(postCommentService.deletePostCommentByPostId(1L)).thenReturn(1);
        when(postFavoriteService.deletePostFavoriteByPostId(1L)).thenReturn(1);
        when(postTagService.deletePostTagRelation(1L)).thenReturn(1);
        when(postService.deletePostByIds(any())).thenReturn(1);

        mockMvc.perform(delete("/post/{postIds}", "1"))
                .andExpect(status().isOk());

        verify(postCommentService, times(1)).deletePostCommentByPostId(1L);
        verify(postFavoriteService, times(1)).deletePostFavoriteByPostId(1L);
        verify(postTagService, times(1)).deletePostTagRelation(1L);
        verify(postService, times(1)).deletePostByIds(any());
    }

    /**
     * 测试导出帖子
     */
    @Test
    void testExportPosts() throws Exception {
        List<Post> posts = new ArrayList<>();
        when(postService.selectPostList(any(Post.class))).thenReturn(posts);

        mockMvc.perform(post("/post/export"))
                .andExpect(status().isOk());

        verify(postService, times(1)).selectPostList(any(Post.class));
    }

    /**
     * 测试收藏帖子
     */
    @Test
    void testFavoritePost() throws Exception {
        Post post = new Post();
        post.setPostId(1L);
        post.setTitle("测试帖子");

        when(postService.selectPostById(1L)).thenReturn(post);
        when(postFavoriteService.selectPostFavoriteByPostIdAndUserId(eq(1L), eq(1L))).thenReturn(null);
        when(postFavoriteService.insertPostFavorite(any())).thenReturn(1);
        when(postService.updatePostFavorites(eq(1L), eq(1))).thenReturn(1);

        mockMvc.perform(post("/post/favorite/{postId}", 1L))
                .andExpect(status().isOk());

        verify(postService, times(1)).selectPostById(1L);
        verify(postFavoriteService, times(1)).insertPostFavorite(any());
    }

    /**
     * 测试获取收藏列表
     */
    @Test
    void testGetFavoriteList() throws Exception {
        List<PostFavorite> favorites = new ArrayList<>();
        when(postFavoriteService.selectPostFavoriteList(any())).thenReturn(favorites);

        mockMvc.perform(get("/post/favorite/list"))
                .andExpect(status().isOk());

        verify(postFavoriteService, times(1)).selectPostFavoriteList(any());
    }

    /**
     * 测试获取待审核帖子列表
     */
    @Test
    void testGetReviewList() throws Exception {
        List<Post> posts = new ArrayList<>();
        when(postService.selectPostList(any())).thenReturn(posts);

        mockMvc.perform(get("/post/review/list"))
                .andExpect(status().isOk());

        verify(postService, times(1)).selectPostList(any());
    }

    /**
     * 测试审核通过帖子
     */
    @Test
    void testReviewPostApprove() throws Exception {
        Post post = new Post();
        post.setPostId(1L);
        post.setStatus("0");

        when(postService.selectPostById(1L)).thenReturn(post);
        when(postService.updatePost(any())).thenReturn(1);

        String requestBody = "{\"action\":\"approve\",\"reason\":\"审核通过\"}";

        mockMvc.perform(put("/post/review/{postId}", 1L)
                .contentType(MediaType.APPLICATION_JSON)
                .content(requestBody))
                .andExpect(status().isOk());

        verify(postService, times(1)).selectPostById(1L);
        verify(postService, times(1)).updatePost(any());
    }

    /**
     * 测试获取举报列表
     */
    @Test
    void testGetReportList() throws Exception {
        List<PostReport> reports = new ArrayList<>();
        when(postReportMapper.selectPostReportList(any())).thenReturn(reports);

        mockMvc.perform(get("/post/report/list"))
                .andExpect(status().isOk());

        verify(postReportMapper, times(1)).selectPostReportList(any());
    }

    /**
     * 测试获取帖子详情（帖子不存在）
     */
    @Test 
    void testGetNonExistentPostInfo() throws Exception {
        when(postService.selectPostById(999L)).thenReturn(null);
        when(postService.updatePostViews(999L)).thenReturn(0);

        // 控制器现在会检查post是否为null并返回错误
        mockMvc.perform(get("/post/{postId}", 999L))
                .andExpect(status().isOk());

        verify(postService, times(1)).selectPostById(999L);
        verify(postService, times(1)).updatePostViews(999L);
    }

    /**
     * 测试编辑不存在的帖子
     */
    @Test
    void testEditNonExistentPost() throws Exception {
        Post post = new Post();
        post.setPostId(999L);
        post.setTitle("不存在的帖子");

        when(postService.updatePost(any(Post.class))).thenReturn(0);
        when(postTagService.deletePostTagRelation(999L)).thenReturn(0);

        mockMvc.perform(put("/post")
                .contentType(MediaType.APPLICATION_JSON)
                .content(objectMapper.writeValueAsString(post)))
                .andExpect(status().isOk());

        verify(postService, times(1)).updatePost(any(Post.class));
    }

    /**
     * 测试删除不存在的帖子
     */
    @Test
    void testDeleteNonExistentPost() throws Exception {
        when(postCommentService.deletePostCommentByPostId(999L)).thenReturn(0);
        when(postFavoriteService.deletePostFavoriteByPostId(999L)).thenReturn(0);
        when(postTagService.deletePostTagRelation(999L)).thenReturn(0);
        when(postService.deletePostByIds(any())).thenReturn(0);

        mockMvc.perform(delete("/post/{postIds}", "999"))
                .andExpect(status().isOk());

        verify(postCommentService, times(1)).deletePostCommentByPostId(999L);
        verify(postFavoriteService, times(1)).deletePostFavoriteByPostId(999L);
        verify(postService, times(1)).deletePostByIds(any());
    }

    /**
     * 测试审核不存在的帖子
     */
    @Test
    void testReviewNonExistentPost() throws Exception {
        when(postService.selectPostById(999L)).thenReturn(null);

        String requestBody = "{\"action\":\"approve\",\"reason\":\"审核通过\"}";

        mockMvc.perform(put("/post/review/{postId}", 999L)
                .contentType(MediaType.APPLICATION_JSON)
                .content(requestBody))
                .andExpect(status().isOk());

        verify(postService, times(1)).selectPostById(999L);
        verify(postService, never()).updatePost(any());
    }

    /**
     * 测试添加带标签的帖子
     */
    @Test
    void testAddPostWithTags() throws Exception {
        Post post = new Post();
        post.setTitle("带标签的帖子");
        post.setContent("内容");
        post.setTags("Java,Spring");
        post.setPostId(1L); // 模拟插入后的ID

        when(postService.insertPost(any(Post.class))).thenReturn(1);
        when(postTagService.selectPostTagList(any(PostTag.class))).thenReturn(new ArrayList<>());
        when(postTagService.insertPostTag(any(PostTag.class))).thenReturn(1);
        when(postTagService.batchInsertPostTagRelation(eq(1L), any())).thenReturn(1);

        mockMvc.perform(post("/post")
                .contentType(MediaType.APPLICATION_JSON)
                .content(objectMapper.writeValueAsString(post)))
                .andExpect(status().isOk());

        verify(postService, times(1)).insertPost(any(Post.class));
    }

    /**
     * 测试批量删除帖子
     */
    @Test
    void testBatchDeletePosts() throws Exception {
        when(postCommentService.deletePostCommentByPostId(1L)).thenReturn(1);
        when(postCommentService.deletePostCommentByPostId(2L)).thenReturn(1);
        when(postFavoriteService.deletePostFavoriteByPostId(1L)).thenReturn(1);
        when(postFavoriteService.deletePostFavoriteByPostId(2L)).thenReturn(1);
        when(postTagService.deletePostTagRelation(1L)).thenReturn(1);
        when(postTagService.deletePostTagRelation(2L)).thenReturn(1);
        when(postService.deletePostByIds(any())).thenReturn(2);

        mockMvc.perform(delete("/post/{postIds}", "1,2"))
                .andExpect(status().isOk());

        verify(postCommentService, times(1)).deletePostCommentByPostId(1L);
        verify(postCommentService, times(1)).deletePostCommentByPostId(2L);
        verify(postFavoriteService, times(1)).deletePostFavoriteByPostId(1L);
        verify(postFavoriteService, times(1)).deletePostFavoriteByPostId(2L);
        verify(postService, times(1)).deletePostByIds(any());
    }
} 