package bounty.service;

import bounty.domain.Bounty;
import bounty.domain.BountySubmission;
import bounty.mapper.BountyMapper;
import bounty.mapper.BountySubmissionMapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.ruoyi.common.utils.SecurityUtils;
import org.mybatis.spring.MyBatisSystemException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Date;

@Service
public class BountySubmissionServiceImpl extends ServiceImpl<BountySubmissionMapper, BountySubmission> implements BountySubmissionService {

    @Autowired
    private BountyMapper bountyMapper; // 注入 BountyMapper 校验悬赏存在性

    @Override
    public boolean saveBountySubmission(BountySubmission bountySubmission) {
        // 校验 1：悬赏 ID 不能为空
        Long bountyId = bountySubmission.getBountyId();
        if (bountyId == null) {
            throw new IllegalArgumentException("悬赏 ID 不能为空");
        }

        // 校验 2：目标悬赏是否存在
        Bounty targetBounty = bountyMapper.selectById(bountyId);
        if (targetBounty == null) {
            throw new IllegalArgumentException("无效的悬赏 ID，未找到对应悬赏");
        }

        // 校验 3：提交内容不能为空
        String content = bountySubmission.getContent();
        if (!StringUtils.hasText(content)) {
            throw new IllegalArgumentException("提交内容不能为空");
        }

        // 自动填充当前用户 ID（需用户已登录）
        Long currentUserId = SecurityUtils.getUserId();
        bountySubmission.setUserId(currentUserId);


        // 调用 MyBatis-Plus 保存方法
        boolean saveResult = this.save(bountySubmission);

        // 新增：保存成功后，更新对应悬赏的状态为1（已回复）
        if (saveResult) {
            targetBounty.setStatus(1); // 将悬赏状态改为1
            bountyMapper.updateById(targetBounty); // 执行更新
        }

        return saveResult;
    }

    @Override
    public IPage<BountySubmission> getBountySubmissionsByPage(Page<BountySubmission> page, Long bountyId) {
        QueryWrapper<BountySubmission> wrapper = new QueryWrapper<>();
        wrapper.eq("bounty_id", bountyId);
        return this.page(page, wrapper);
    }

    private static final Logger log = LoggerFactory.getLogger(BountySubmissionServiceImpl.class);

    @Override
    public boolean adoptSubmission(Long submissionId) {
        log.info("【采纳流程开始】submissionId={}, 当前时间={}", submissionId, new Date());

        try {
            // 参数校验
            if (submissionId == null || submissionId <= 0) {
                log.warn("【参数校验失败】submissionId为空或无效");
                throw new IllegalArgumentException("无效的回复ID");
            }

            // 获取回复详情
            BountySubmission submission = this.getById(submissionId);
            if (submission == null) {
                log.warn("【回复不存在】submissionId={}", submissionId);
                throw new IllegalArgumentException("无效的回复ID");
            }

            if (submission.getId() == null || submission.getBountyId() == null) {
                log.warn("【无效提交数据】submissionId={}", submissionId);
                throw new IllegalArgumentException("提交数据不完整");
            }


            log.debug("【原始回复数据】{}", submission);

            // 获取关联悬赏
            Bounty bounty = bountyMapper.selectById(submission.getBountyId());

            if (bounty == null) {
                log.warn("【关联悬赏不存在】bountyId={}", submission.getBountyId());
                throw new IllegalArgumentException("关联悬赏不存在");
            }

            // 用户权限校验
            Long currentUserId = SecurityUtils.getUserId();
            if (currentUserId == null) {
                log.warn("【用户未登录】");
                throw new IllegalArgumentException("用户未登录");
            }


            if (!bounty.getCreator_id().equals(currentUserId)) {
                log.warn("【权限不足】用户ID={} 试图访问悬赏ID={}", currentUserId, bounty.getId());
                throw new IllegalArgumentException("无权操作");
            }

            // 状态校验
            if (submission.getStatus() == 1) {
                log.warn("【重复采纳】submissionId={}", submissionId);
                throw new IllegalArgumentException("该回复已被采纳");
            }

            log.info("【更新前对象】{}", submission);

            // 执行更新
            submission.setStatus(1);
            boolean submissionResult = this.updateById(submission);
            log.info("【更新后对象】{}", this.getById(submissionId));


            log.info("【操作结果】submissionResult={}, 新状态={}", submissionResult, submission.getStatus());

            return submissionResult;
        } catch (MyBatisSystemException e) {
            // 显式捕获 MyBatis 异常
            log.error("【MyBatis 异常】submissionId={}, rootCause={}", submissionId, e.getRootCause());
            log.error("【异常详情】", e); // 打印完整堆栈
            throw new RuntimeException("MyBatis 错误: " + e.getRootCause(), e);

    } catch (IllegalArgumentException e) {
            log.warn("【参数异常】submissionId={}, message={}", submissionId, e.getMessage());
            throw e;
        } catch (Exception e) {
            log.error("【采纳异常】submissionId={}, 错误={}", submissionId, e.getMessage(), e);
            log.error("【未分类异常】submissionId={}, exceptionClass={}", submissionId, e.getClass().getName());
            log.error("【异常详情】", e); // 打印完整堆栈
            throw e;
        }
    }
}
