// src/services/bounty.ts
import { request } from '@umijs/max';

// 查询悬赏列表
export async function getBountyList(params?: Record<string, any>) {
  return request('/api/bounties', {
    method: 'GET',
    headers: {
      'Content-Type': 'application/json;charset=UTF-8',
    },
    params,
  });
}

export async function getBountyDetail(id: number | string) {
  return request(`/api/bounties/${id}`, {
    method: 'GET',
    headers: {
      'Content-Type': 'application/json;charset=UTF-8',
    },
  }).then(data => ({ code: 200, data })); // ✅ 自动包装成标准结构
}

// 新增：发布悬赏
export async function publishBounty(params: {
  title: string;
  description: string;
  reward: number;
  deadline: string;
}) {
  return request('/api/bounties/publish', {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json;charset=UTF-8',
    },
    data: {
      ...params,
      status: 0
    },
  });
}

// 新增：上传附件接口
export async function uploadBountyAttachment(file: File) {
  const formData = new FormData();
  formData.append('file', file);

  return request('/api/bounty-submissions/upload', {
    method: 'POST',
    data: formData,
    // ✅ 移除手动设置 Content-Type
    // headers: {
    //   'Content-Type': 'multipart/form-data', // ❌ 浏览器会自动设置 boundary
    // },
  });
}

// 修改：提交悬赏回复（不再处理文件）
// src/services/bounty/bounty.ts
// src/services/bounty/bounty.ts
// src/services/bounty/bounty.ts
export async function submitBountyReply(params: {
  bountyId: number;
  content: string;
  attachment?: string;
  file?: File;
}) {
  console.log('【提交请求】/api/bounty-submissions 参数:', params);

  const formData = new FormData();
  formData.append('submission', new Blob([JSON.stringify(params)], { type: 'application/json' }));
  if (params.file) {
    formData.append('file', params.file);
  }

  return request('/api/bounty-submissions', {
    method: 'POST',
    data: formData,
  }).then(res => {
    console.log('【接口响应】/api/bounty-submissions:', res);
    // ✅ 确保返回统一结构
    return {
      code: res.code || (res ? 200 : 500),
      data: res.data || res,
      msg: res.message || '操作成功',
    };
  });
}


export async function downloadAttachment(attachmentPath: string): Promise<Blob> {
  // ✅ 提取文件名
  const filename = attachmentPath.split('/').pop() || 'file';

  return request(`/api/bounty-submissions/download`, {
    method: 'GET',
    params: {
      filename, // ✅ 只传递文件名
    },
    responseType: 'blob',
  });
}

export async function getProfile() {
  return request('/api/system/user/profile', {
    method: 'GET',
    headers: {
      'Content-Type': 'application/json;charset=UTF-8',
    },
  });
}

export async function adoptSubmission(submissionId: number) {
  return request(`/api/bounty-submissions/${submissionId}/adopt`, {
    method: 'PUT',
    headers: {
      'Content-Type': 'application/json;charset=UTF-8',
    },
  });
}
