import * as bountyService from '../../src/services/bounty/bounty';

// Mock request module
jest.mock('@umijs/max', () => ({
  request: jest.fn(),
}));

const { request } = require('@umijs/max');

describe('Bounty Service', () => {
  beforeEach(() => {
    jest.clearAllMocks();
  });

  // ✅ 悬赏列表相关测试
  describe('getBountyList', () => {
    it('应该成功获取悬赏列表', async () => {
      const mockResponse = {
        code: 200,
        data: {
          records: [
            {
              id: 1,
              title: '测试悬赏',
              creator_id: 1,
              reward: 100,
              deadline: '2023-12-31',
              status: 0,
              submissionsCount: 0
            }
          ],
          total: 1
        }
      };

      (request as jest.Mock).mockResolvedValue(mockResponse);

      const result = await bountyService.getBountyList({ pageNum: 1, pageSize: 10 });

      expect(result).toEqual(mockResponse);
      expect(request).toHaveBeenCalledWith('/api/bounties', {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json;charset=UTF-8',
        },
        params: { pageNum: 1, pageSize: 10 },
      });
    });
  });


  // ✅ 发布悬赏测试
  describe('publishBounty', () => {
    it('应该成功发布悬赏', async () => {
      const mockResponse = {
        code: 200,
        msg: '发布成功'
      };

      (request as jest.Mock).mockResolvedValue(mockResponse);

      const postData = {
        title: '新悬赏标题',
        description: '新悬赏描述',
        reward: 100,
        deadline: '2023-12-31T23:59:59'
      };

      const result = await bountyService.publishBounty(postData);

      expect(result).toEqual(mockResponse);
      expect(request).toHaveBeenCalledWith('/api/bounties/publish', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json;charset=UTF-8',
        },
        data: {
          ...postData,
          status: 0
        },
      });
    });
  });

  // ✅ 提交回复测试
  describe('submitBountyReply', () => {
    it('应该成功提交回复', async () => {
      const mockResponse = {
        code: 200,
        data: {
          id: 1
        },
        msg: '操作成功'
      };

      (request as jest.Mock).mockResolvedValue(mockResponse);

      const params = {
        bountyId: 1,
        content: '这是一条回复内容',
        file: new File(['test'], 'test.txt') as File
      };

      const result = await bountyService.submitBountyReply(params);

      expect(result).toEqual(mockResponse);
      expect(request).toHaveBeenCalledWith('/api/bounty-submissions', {
        method: 'POST',
        data: expect.any(FormData)
      });
    });

    it('应该处理空回复内容', async () => {
      const mockResponse = {
        code: 400,
        data: {
          msg: '内容不能为空'
        },
        msg: '操作成功'
      };

      (request as jest.Mock).mockResolvedValue(mockResponse);

      const result = await bountyService.submitBountyReply({
        bountyId: 1,
        content: ''
      });

      expect(result).toEqual(mockResponse);
    });
  });

  // ✅ 错误处理测试
  describe('error handling', () => {
    it('应该处理网络错误', async () => {
      const error = new Error('Network Error');
      (request as jest.Mock).mockRejectedValue(error);

      await expect(bountyService.getBountyList()).rejects.toThrow('Network Error');
    });

    it('应该处理服务器错误响应', async () => {
      const mockResponse = { code: 500, msg: '服务器内部错误' };
      (request as jest.Mock).mockResolvedValue(mockResponse);

      const result = await bountyService.getBountyList();

      expect(result).toEqual(mockResponse);
    });
  });
});
