import React, { useEffect, useState } from "react";
import { styled } from "@mui/material/styles";

import { getCategories, getTorrentList } from "../../services/bt/index";
import {
Box,
Button,
Card,
CardContent,
Chip,
CircularProgress,
Container,
MenuItem,
Pagination,
Select,
Typography,
} from "@mui/material";

// 星空背景
const StarBg = styled("div")({
minHeight: "100vh",
width: "auto",
background: "radial-gradient(ellipse at bottom, #1b2735 0%, #090a0f 100%)",
overflow: "auto",
position: "relative",
top: 0,
left: 0,
"&:before, &:after": {
    content: '""',
    position: "absolute",
    top: 0,
    left: 0,
    width: "100%",
    height: "100%",
    pointerEvents: "none",
},
"&:before": {
    background: `
        radial-gradient(1px 1px at 20% 30%, rgba(255,255,255,0.8), rgba(255,255,255,0)),
        radial-gradient(1px 1px at 40% 70%, rgba(255,255,255,0.8), rgba(255,255,255,0)),
        radial-gradient(1.5px 1.5px at 60% 20%, rgba(255,255,255,0.9), rgba(255,255,255,0)),
        radial-gradient(1.5px 1.5px at 80% 90%, rgba(255,255,255,0.9), rgba(255,255,255,0))
    `,
    backgroundRepeat: "repeat",
    backgroundSize: "200px 200px",
    animation: "twinkle 10s infinite ease-in-out",
},
"&:after": {
    background: `
        radial-gradient(1px 1px at 70% 40%, rgba(255,255,255,0.7), rgba(255,255,255,0)),
        radial-gradient(1.2px 1.2px at 10% 80%, rgba(255,255,255,0.7), rgba(255,255,255,0)),
        radial-gradient(1.5px 1.5px at 30% 60%, rgba(255,255,255,0.8), rgba(255,255,255,0))
    `,
    backgroundRepeat: "repeat",
    backgroundSize: "300px 300px",
    animation: "twinkle 15s infinite 5s ease-in-out",
},
"@keyframes twinkle": {
    "0%, 100%": { opacity: 0.3 },
    "50%": { opacity: 0.8 },
}
});

const CategoryBar = styled(Box)({
display: "flex",
gap: 12,
alignItems: "center",
marginBottom: 24,
flexWrap: "wrap",
});

const TorrentCard = styled(Card)({
background: "rgba(30, 41, 59, 0.85)",
color: "#fff",
borderRadius: 16,
boxShadow: "0 4px 24px 0 rgba(0,0,0,0.4)",
border: "1px solid #334155",
transition: "transform 0.2s",
"&:hover": {
    transform: "scale(1.025)",
    boxShadow: "0 8px 32px 0 #0ea5e9",
},
});

const sortOptions = [
{ label: "最新", value: { sortField: "createTime", sortDirection: "desc" } },
{ label: "下载量", value: { sortField: "completions", sortDirection: "desc" } },
{ label: "推荐", value: { sortField: "seeders", sortDirection: "desc" } },
];

const statusMap: Record<number, string> = {
0: '审核中',
1: '已发布',
2: '审核不通过',
3: '已上架修改重审中',
10: '已下架',
};

const PAGE_SIZE = 20;

const TorrentAuditListPage: React.FC = () => {
const [categories, setCategories] = useState<
    { id: number; name: string; remark?: string | null; type?: number | null }[]
>([]);
const [selectedCat, setSelectedCat] = useState<number | null>(null);
const [sort, setSort] = useState(sortOptions[0].value);
const [torrents, setTorrents] = useState<any[]>([]);
const [loading, setLoading] = useState(false);
const [page, setPage] = useState(1);
const [total, setTotal] = useState(0);

useEffect(() => {
    getCategories().then((res) => {
        setCategories(res.data || []);
    });
}, []);

useEffect(() => {
    setLoading(true);
    getTorrentList({
        category: selectedCat !== null ? String(selectedCat) : undefined,
        sortField: sort.sortField,
        sortDirection: sort.sortDirection,
        pageNum: page,
        pageSize: PAGE_SIZE,
    })
        .then((res) => {
            setTorrents(res.data || []);
            setTotal(res.page?.total || 0);
        })
        .finally(() => setLoading(false));
}, [selectedCat, sort, page]);

return (
    <StarBg>
        <Container maxWidth="lg" sx={{ pt: 6, pb: 6, position: "relative" }}>
            <Typography
                variant="h3"
                sx={{
                    color: "#fff",
                    fontWeight: 700,
                    mb: 3,
                    letterSpacing: 2,
                    textShadow: "0 2px 16px #0ea5e9",
                }}
            >
                ThunderHub 种子审核
            </Typography>
            <CategoryBar>
                <Chip
                    label="全部"
                    color={selectedCat === null ? "primary" : "default"}
                    onClick={() => {
                        setSelectedCat(null);
                        setPage(1);
                    }}
                    sx={{
                        fontWeight: 600,
                        fontSize: 16,
                        color: selectedCat === null ? undefined : "#fff",
                    }}
                />
                {categories.map((cat) => (
                    <Chip
                        key={cat.id}
                        label={cat.name}
                        color={selectedCat === cat.id ? "primary" : "default"}
                        onClick={() => {
                            setSelectedCat(cat.id);
                            setPage(1);
                        }}
                        sx={{
                            fontWeight: 600,
                            fontSize: 16,
                            color: selectedCat === cat.id ? undefined : "#fff",
                        }}
                    />
                ))}
                <Box sx={{ flex: 1 }} />
                <Select
                    size="small"
                    value={JSON.stringify(sort)}
                    onChange={(e) => {
                        setSort(JSON.parse(e.target.value));
                        setPage(1);
                    }}
                    sx={{
                        color: "#fff",
                        background: "#1e293b",
                        borderRadius: 2,
                        ".MuiOutlinedInput-notchedOutline": { border: 0 },
                        minWidth: 120,
                    }}
                >
                    {sortOptions.map((opt) => (
                        <MenuItem key={opt.label} value={JSON.stringify(opt.value)}>
                            {opt.label}
                        </MenuItem>
                    ))}
                </Select>
            </CategoryBar>
            {loading ? (
                <Box sx={{ display: "flex", justifyContent: "center", mt: 8 }}>
                    <CircularProgress color="info" />
                </Box>
            ) : (
                <>
                    <Box
                        sx={{
                            display: "flex",
                            flexWrap: "wrap",
                            gap: 3,
                            justifyContent: { xs: "center", md: "flex-start" },
                        }}
                    >
                        {torrents.map((torrent) => (
                            <Box
                                key={torrent.id}
                                sx={{
                                    flex: "1 1 320px",
                                    maxWidth: { xs: "100%", sm: "48%", md: "32%" },
                                    minWidth: 300,
                                    mb: 3,
                                    display: "flex",
                                    cursor: "pointer",
                                }}
                                onClick={() => window.open(`/torrent-audit/${torrent.id}`, "_self")}
                            >
                                <TorrentCard sx={{ width: "100%" }}>
                                    <CardContent>
                                        <Typography
                                            variant="h6"
                                            sx={{
                                                color: "#38bdf8",
                                                fontWeight: 700,
                                                mb: 1,
                                                textOverflow: "ellipsis",
                                                overflow: "hidden",
                                                whiteSpace: "nowrap",
                                            }}
                                            title={torrent.title}
                                        >
                                            {torrent.title}
                                        </Typography>
                                        <Box
                                            sx={{
                                                display: "flex",
                                                gap: 1,
                                                alignItems: "center",
                                                mb: 1,
                                                flexWrap: "wrap",
                                            }}
                                        >
                                            <Chip
                                                size="small"
                                                label={
                                                    categories.find((c) => c.id === torrent.category)?.name ||
                                                    "未知"
                                                }
                                                sx={{
                                                    background: "#0ea5e9",
                                                    color: "#fff",
                                                    fontWeight: 600,
                                                }}
                                            />
                                            {torrent.free === "1" && (
                                                <Chip
                                                    size="small"
                                                    label="促销"
                                                    sx={{
                                                        background: "#fbbf24",
                                                        color: "#1e293b",
                                                        fontWeight: 600,
                                                    }}
                                                />
                                            )}
                                            {typeof torrent.status !== "undefined" && (
                                                <Chip
                                                    size="small"
                                                    label={statusMap[torrent.status] || "未知状态"}
                                                    sx={{
                                                        background: "#64748b",
                                                        color: "#fff",
                                                        fontWeight: 600,
                                                    }}
                                                />
                                            )}
                                        </Box>
                                        <Typography variant="body2" sx={{ color: "#cbd5e1", mb: 1 }}>
                                            上传时间：{torrent.createTime}
                                        </Typography>
                                        <Box sx={{ display: "flex", gap: 2, mt: 1 }}>
                                            <Typography variant="body2" sx={{ color: "#38bdf8" }}>
                                                做种：{torrent.seeders}
                                            </Typography>
                                            <Typography variant="body2" sx={{ color: "#f472b6" }}>
                                                下载量：{torrent.completions}
                                            </Typography>
                                            <Typography variant="body2" sx={{ color: "#fbbf24" }}>
                                                下载中：{torrent.leechers}
                                            </Typography>
                                        </Box>
                                    </CardContent>
                                </TorrentCard>
                            </Box>
                        ))}
                    </Box>
                    <Box sx={{ display: "flex", justifyContent: "center", mt: 4 }}>
                        <Pagination
                            count={Math.ceil(total / PAGE_SIZE)}
                            page={page}
                            onChange={(_, v) => setPage(v)}
                            color="primary"
                            sx={{
                                ".MuiPaginationItem-root": {
                                    color: "#fff",
                                    background: "#1e293b",
                                    border: "1px solid #334155",
                                },
                                ".Mui-selected": {
                                    background: "#0ea5e9 !important",
                                },
                            }}
                        />
                    </Box>
                </>
            )}
        </Container>
    </StarBg>
);
};

export default TorrentAuditListPage;