import { request } from '@umijs/max';

export interface PostListParams {
  pageNum?: number;
  pageSize?: number;
  title?: string;
  status?: string;
  tags?: string;
}

export interface PostDetailResponse {
  post: API.Post.PostInfo;
  tags: API.Post.PostTag[];
  comments: any[];
  authorPosts: API.Post.PostInfo[];
  similarPosts: API.Post.PostInfo[];
  recommendedPosts: API.Post.PostInfo[];
  favorited: boolean;
}

export interface CommentAddParams {
  postId: number;
  content: string;
  parentId?: number;
}

// 获取帖子列表
export async function getPostList(params: PostListParams) {
    console.log('getPostList', params);    
  return request<API.TableDataInfo>('/api/post-center/list', {
    method: 'GET',
    params,
  });
}

// 获取帖子详情
export async function getPostDetail(postId: number) {
    console.log('getPostDetail', postId);
  return request<API.AjaxResult<PostDetailResponse>>(`/api/post-center/${postId}`, {
    method: 'GET',
  });
}

// 添加评论
export async function addComment(data: CommentAddParams) {
    console.log('addComment', data);
  return request<API.AjaxResult>('/api/post-center/comment', {
    method: 'POST',
    data,
  });
}

// 收藏/取消收藏帖子
export async function toggleFavorite(postId: number, favorite: boolean) {
    console.log('toggleFavorite', postId, favorite);
  return request<API.AjaxResult>(`/api/post-center/favorite/${postId}`, {
    method: 'POST',
    params: { favorite },
  });
}

// 获取热门标签
export async function getHotTags(): Promise<API.AjaxResult<API.Post.PostTag[]>> {
  return request('/api/post-center/tags/hot', {
    method: 'GET',
  });
}

// 根据标签获取帖子
export async function getPostsByTag(tagId: number) {
    console.log('getPostsByTag', tagId);
  return request<API.TableDataInfo>(`/api/post-center/bytag/${tagId}`, {
    method: 'GET',
  });
}

// 发布帖子
export async function publishPost(data: {
  title: string;
  content: string;
  summary: string;
  tags: string;
  promotionPlan?: number;
}): Promise<API.AjaxResult> {
  return request('/api/post-center/publish', {
    method: 'POST',
    data,
  });
}

// 获取我的帖子列表
export async function getMyPosts(params: {
  pageNum?: number;
  pageSize?: number;
}): Promise<API.TableDataInfo<API.Post.PostInfo>> {
  return request('/api/post-center/my-posts', {
    method: 'GET',
    params,
  });
}

// 获取我的收藏列表
export async function getMyFavorites(params: {
  pageNum?: number;
  pageSize?: number;
}): Promise<API.TableDataInfo<API.Post.PostInfo>> {
  return request('/api/post-center/my-favorites', {
    method: 'GET',
    params,
  });
}

// 更新帖子
export async function updatePost(data: API.Post.PostInfo): Promise<API.AjaxResult> {
  return request('/api/post-center/update', {
    method: 'PUT',
    data,
  });
}

// 删除帖子
export async function deletePost(postId: number): Promise<API.AjaxResult> {
  return request(`/api/post-center/delete/${postId}`, {
    method: 'DELETE',
  });
}

// 获取可用标签列表（用于下拉选择）
export async function getAvailableTags(): Promise<API.AjaxResult<API.Post.PostTag[]>> {
  return request('/api/post-center/tags/available', {
    method: 'GET',
  });
}

// 上传图片
export async function uploadImage(file: FormData): Promise<API.AjaxResult<{url: string}>> {
  return request('/api/post-center/upload', {
    method: 'POST',
    data: file,
  });
} 

// 删除图片
export async function deleteImage(filename: string): Promise<API.AjaxResult> {
  return request('/api/post-center/upload', {
    method: 'DELETE',
    params: { filename },
  });
}

// 获取推广计划列表
export async function getPromotionPlans(): Promise<API.AjaxResult<any[]>> {
  return request('/api/post-center/promotion-plans', {
    method: 'GET',
  });
}

// 创建支付记录
export async function createPayment(data: {
  postId: number;
  planId: number;
  amount: number;
}): Promise<API.AjaxResult> {
  return request('/api/post-center/payment', {
    method: 'POST',
    data,
  });
}

// 点赞/取消点赞帖子
export async function toggleLike(postId: number, like: boolean) {
  console.log('toggleLike', postId, like);
  return request<API.AjaxResult>(`/api/post-center/like/${postId}`, {
    method: 'POST',
    params: { like },
  });
}

// 获取推广帖子列表（用于轮播展示）
export async function getPromotionPosts(): Promise<API.AjaxResult<API.Post.PostInfo[]>> {
  return request('/api/post-center/promotion', {
    method: 'GET',
  });
}

// 点赞/取消点赞评论
export async function toggleCommentLike(commentId: number, like: boolean) {
  console.log('toggleCommentLike', commentId, like);
  return request<API.AjaxResult>(`/api/post-center/comment/like/${commentId}`, {
    method: 'POST',
    params: { like },
  });
}

// 获取待审核帖子列表
export async function getReviewPosts(params: PostListParams): Promise<API.TableDataInfo<API.Post.PostInfo>> {
  return request('/api/post/review/list', {
    method: 'GET',
    params,
  });
}

// 审核帖子（通过/拒绝）
export async function reviewPost(postId: number, action: 'approve' | 'reject', reason?: string): Promise<API.AjaxResult> {
  return request(`/api/post/review/${postId}`, {
    method: 'PUT',
    data: { action, reason },
  });
}

// 强制下架帖子
export async function takeDownPost(postId: number, reason?: string): Promise<API.AjaxResult> {
  return request(`/api/post/takedown/${postId}`, {
    method: 'PUT',
    data: { reason },
  });
}

// 检查帖子推广状态
export async function getPromotionStatus(postId: number): Promise<API.AjaxResult> {
  return request(`/api/post-center/promotion-status/${postId}`, {
    method: 'GET',
  });
}

// 确认支付成功
export async function confirmPayment(paymentId: number): Promise<API.AjaxResult> {
  return request(`/api/post-center/payment/confirm/${paymentId}`, {
    method: 'POST',
  });
}

// 取消支付
export async function cancelPayment(paymentId: number): Promise<API.AjaxResult> {
  return request(`/api/post-center/payment/cancel/${paymentId}`, {
    method: 'POST',
  });
}

// 举报帖子
export async function reportPost(postId: number, reason: string): Promise<API.AjaxResult> {
  return request(`/api/post-center/report/${postId}`, {
    method: 'POST',
    data: { reason },
  });
}

// 获取举报列表
export async function getReportList(params: PostListParams): Promise<API.TableDataInfo> {
  return request('/api/post/report/list', {
    method: 'GET',
    params,
  });
}

// 处理举报
export async function handleReport(reportId: number, action: 'approve' | 'reject', postId: number, reason?: string): Promise<API.AjaxResult> {
  return request(`/api/post/report/handle/${reportId}`, {
    method: 'PUT',
    data: { action, postId, reason },
  });
} 