import React, { useState } from 'react'
import { connect } from 'dva'
import { history } from 'umi'
import { Form, Input, Button, message } from 'antd'
import { UserOutlined, LockOutlined } from '@ant-design/icons'
import styles from './index.less'

const RegisterPage = ({ dispatch, register }) => {
  const [form] = Form.useForm()
  
  const onFinish = values => {
    dispatch({
      type: 'register/submit',
      payload: values
    }).then(() => {
      if (!register.error) {
        message.success('注册成功')
        history.push('/user/login')
      } else {
        message.error(register.error)
      }
    })
  }
  
  return (
    <div className={styles.container}>
      <div className={styles.content}>
        <div className={styles.top}>
          <div className={styles.header}>
            <span className={styles.title}>用户注册</span>
          </div>
        </div>
        
        <div className={styles.main}>
          <Form
            form={form}
            name="register"
            onFinish={onFinish}
            scrollToFirstError
          >
            <Form.Item
              name="username"
              rules={[
                { required: true, message: '请输入用户名!' },
                { min: 4, message: '用户名至少4个字符' },
                { max: 20, message: '用户名最多20个字符' },
                { pattern: /^[a-zA-Z0-9_]+$/, message: '用户名只能包含字母、数字和下划线' }
              ]}
            >
              <Input
                prefix={<UserOutlined className={styles.prefixIcon} />}
                placeholder="用户名"
                size="large"
              />
            </Form.Item>
            
            <Form.Item
              name="password"
              rules={[
                { required: true, message: '请输入密码!' },
                { min: 6, message: '密码至少6个字符' },
                { max: 20, message: '密码最多20个字符' }
              ]}
            >
              <Input.Password
                prefix={<LockOutlined className={styles.prefixIcon} />}
                type="password"
                placeholder="密码"
                size="large"
              />
            </Form.Item>
            
            <Form.Item
              name="confirmPassword"
              dependencies={['password']}
              rules={[
                { required: true, message: '请确认密码!' },
                ({ getFieldValue }) => ({
                  validator(_, value) {
                    if (!value || getFieldValue('password') === value) {
                      return Promise.resolve()
                    }
                    return Promise.reject(new Error('两次输入的密码不一致!'))
                  }
                })
              ]}
            >
              <Input.Password
                prefix={<LockOutlined className={styles.prefixIcon} />}
                type="password"
                placeholder="确认密码"
                size="large"
              />
            </Form.Item>
            
            <Form.Item>
              <Button
                type="primary"
                htmlType="submit"
                size="large"
                className={styles.submit}
                loading={register.submitting}
                block
              >
                注册
              </Button>
            </Form.Item>
          </Form>
        </div>
      </div>
    </div>
  )
}

export default connect(({ register }) => ({ register }))(RegisterPage)