import React from 'react';
import { Form, Input, InputNumber, DatePicker, Button, message } from 'antd';
import axios from 'axios';
import { publishBounty } from '@/services/bounty/bounty'; // 新增 axios 导入

interface BountyPublishProps {
  onSuccess?: () => void; // 提交成功回调
  onCancel?: () => void;  // 取消操作回调
}



const BountyPublish: React.FC<BountyPublishProps> = ({ onSuccess, onCancel }) => {
  const [form] = Form.useForm();

  // ✅ 替换 axios 请求为服务方法调用
  const handleSubmit = async (values: {
    title: string;
    description: string;
    reward: number;
    deadline: string;
  }) => {
    try {
      // 使用服务层方法
      const res = await publishBounty(values);

      if (res) {
        //message.success('悬赏发布成功！');
        form.resetFields();
        onSuccess?.();
      } else {
        message.error(res.msg || '发布失败，请重试');
      }
    } catch (err) {
      console.error('发布失败:', err);
      message.error('网络请求失败，请检查网络');
    }
  };


  return (
    <div className="page-container">
      <h2>发布新悬赏</h2>
      <Form
        form={form}
        layout="vertical"
        onFinish={handleSubmit}
        requiredMark="optional"
        style={{ maxWidth: 600 }}
      >
        {/* 标题 */}
        <Form.Item
          name="title"
          label="悬赏标题"
          rules={[{ required: true, message: '请输入悬赏标题' }]}
        >
          <Input placeholder="请输入悬赏标题" />
        </Form.Item>

        {/* 描述 */}
        <Form.Item
          name="description"
          label="悬赏描述"
          rules={[{ required: true, message: '请输入悬赏描述' }]}
        >
          <Input.TextArea rows={4} placeholder="请输入悬赏描述" />
        </Form.Item>

        {/* 奖励 */}
        <Form.Item
          name="reward"
          label="悬赏奖励"
          rules={[
            { required: true, message: '请输入奖励数值' },
            { type: 'number', min: 1, message: '奖励必须大于0' },
          ]}
        >
          <InputNumber min={1} placeholder="请输入奖励数值" style={{ width: '100%' }} />
        </Form.Item>

        {/* 截止时间 */}
        <Form.Item
          name="deadline"
          label="截止时间"
          rules={[{ required: true, message: '请选择截止时间' }]}
        >
          <DatePicker
            showTime
            format="YYYY-MM-DD HH:mm:ss"
            placeholder="选择截止时间"
            style={{ width: '100%' }}
          />
        </Form.Item>

        {/* 提交按钮 */}
        <Form.Item>
          <Button type="primary" htmlType="submit">
            发布悬赏
          </Button>
          <Button
            type="default"
            onClick={() => {
              form.resetFields();
              onCancel?.(); // 🔥 取消时触发回调
            }}
            style={{ marginLeft: 16 }}
          >
            重置
          </Button>
        </Form.Item>
      </Form>
    </div>
  );
};

export default BountyPublish;
