package com.example.myproject.controller;

import com.example.myproject.controller.GroupController;
import com.example.myproject.entity.Group;
import com.example.myproject.entity.GroupComments;
import com.example.myproject.service.GroupService;
import com.example.myproject.utils.Result;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.springframework.http.ResponseEntity;

import java.util.HashMap;
import java.util.Map;

import static org.mockito.Mockito.*;
import static org.junit.jupiter.api.Assertions.*;


class GroupControllerTest {

    @InjectMocks
    private GroupController groupController;

    @Mock
    private GroupService groupService;

    @BeforeEach
    void setup() {
        MockitoAnnotations.openMocks(this);
    }

    // 测试创建小组

    @Test
    void createGroupTest() {
        Group groupRequest = new Group();
        groupRequest.setGroupName("Test Group");
        groupRequest.setDescription("This is a test group");

        // 模拟服务层的返回值
        Map<String, Object> mockResponse = new HashMap<>();
        mockResponse.put("group_id", 1L);
        mockResponse.put("message", "兴趣小组创建成功");

        ResponseEntity<Map<String, Object>> mockResponseEntity = ResponseEntity.ok(mockResponse);
        when(groupService.createGroup(groupRequest)).thenReturn(mockResponseEntity);

        // 调用控制器的方法
        ResponseEntity<?> responseEntity = groupController.createGroup(groupRequest);
        Map<String, Object> resultMap = (Map<String, Object>) responseEntity.getBody();
        Result<Map<String, Object>> result = Result.success(resultMap, "兴趣小组创建成功");

        // 验证返回的结果
        assertEquals("200", result.getCode());
        assertEquals("兴趣小组创建成功", result.getMsg());
        assertEquals(1L, result.getData().get("group_id"));
    }


    // 测试加入小组接口
    @Test
    void testJoinGroup() {
        Long groupId = 1L;
        Long userId = 1L;

        // 模拟服务层的返回值
        Map<String, Object> mockResponse = new HashMap<>();
        mockResponse.put("status", "success");
        mockResponse.put("message", "成功加入小组");
        when(groupService.joinGroup(groupId, userId)).thenReturn(ResponseEntity.ok(mockResponse));

        // 调用控制器方法
        Map<String, Long> requestBody = Map.of("user_id", userId);
        ResponseEntity<Map<String, Object>> response = groupController.joinGroup(groupId, requestBody);

        // 验证返回的结果
        assertEquals(200, response.getStatusCodeValue());
        assertEquals("success", response.getBody().get("status"));
        assertEquals("成功加入小组", response.getBody().get("message"));
    }

    // 测试退出小组接口
    @Test
    void testLeaveGroup() {
        Long groupId = 1L;
        Long userId = 1L;

        // 模拟服务层的返回值
        Map<String, Object> mockResponse = new HashMap<>();
        mockResponse.put("status", "success");
        mockResponse.put("message", "成功退出小组");
        when(groupService.leaveGroup(groupId, userId)).thenReturn(ResponseEntity.ok(mockResponse));

        // 调用控制器方法
        Map<String, Long> requestBody = Map.of("user_id", userId);
        ResponseEntity<Map<String, Object>> response = groupController.leaveGroup(groupId, requestBody);

        // 验证返回的结果
        assertEquals(200, response.getStatusCodeValue());
        assertEquals("success", response.getBody().get("status"));
        assertEquals("成功退出小组", response.getBody().get("message"));
    }

    // 测试获取小组成员接口
    @Test
    void testGetGroupMembers() {
        Long groupId = 1L;

        // 模拟服务层的返回值
        Map<String, Object> mockResponse = new HashMap<>();
        mockResponse.put("status", "success");
        mockResponse.put("members", "mock_members");
        when(groupService.getGroupMembers(groupId)).thenReturn(ResponseEntity.ok(mockResponse));

        // 调用控制器方法
        ResponseEntity<Map<String, Object>> response = groupController.getGroupMembers(groupId);

        // 验证返回的结果
        assertEquals(200, response.getStatusCodeValue());
        assertEquals("success", response.getBody().get("status"));
        assertNotNull(response.getBody().get("members"));
    }

    // 测试发布帖子
    @Test
    void createPostTest() {
        Long groupId = 1L;
        Long userId = 2L;
        String postContent = "Test post content";
        String title = "Test post title";

        // 模拟服务层的返回值
        Map<String, Object> mockResponse = new HashMap<>();
        mockResponse.put("post_id", 1L);
        mockResponse.put("message", "帖子创建成功");

        when(groupService.createPost(groupId, userId, postContent, title, null)).thenReturn(mockResponse);

        // 调用控制器的方法
        Map<String, Object> result = groupController.createPost(groupId, userId, postContent, title, null).getBody();

        // 验证返回的结果
        assertEquals("帖子创建成功", result.get("message"));
        assertEquals(1L, result.get("post_id"));
    }

    // 测试点赞帖子
    @Test
    void likePostTest() {
        Long postId = 1L;

        // 模拟服务层的返回值
        Map<String, Object> mockResponse = new HashMap<>();
        mockResponse.put("status", "success");
        mockResponse.put("message", "点赞成功");

        when(groupService.likePost(postId)).thenReturn(mockResponse);

        // 调用控制器的方法
        Map<String, Object> resultMap = groupController.likePost(postId).getBody();

        // 验证返回的结果
        assertEquals("点赞成功", resultMap.get("message"));
    }

    // 测试取消点赞
    @Test
    void unlikePostTest() {
        Long postId = 1L;

        // 模拟服务层的返回值
        Map<String, Object> mockResponse = new HashMap<>();
        mockResponse.put("status", "success");
        mockResponse.put("message", "取消点赞成功");

        when(groupService.unlikePost(postId)).thenReturn(mockResponse);

        // 调用控制器的方法
        Map<String, Object> resultMap = groupController.unlikePost(postId).getBody();

        // 验证返回的结果
        assertEquals("取消点赞成功", resultMap.get("message"));
    }

    // 测试添加评论
    @Test
    void addCommentTest() {
        Long postId = 1L;
        GroupComments comment = new GroupComments();
        comment.setUserId(2L);
        comment.setContent("This is a comment");

        // 模拟服务层的返回值
        Map<String, Object> mockResponse = new HashMap<>();
        mockResponse.put("status", "success");
        mockResponse.put("message", "评论成功");

        when(groupService.addComment(postId, comment)).thenReturn(mockResponse);

        // 调用控制器的方法
        Map<String, Object> resultMap = groupController.addComment(postId, comment).getBody();

        // 验证返回的结果
        assertEquals("评论成功", resultMap.get("message"));
    }
}
