package com.example.myproject.controller;

import com.example.myproject.entity.Post;
import com.example.myproject.service.PostService;
import com.example.myproject.utils.Result;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.*;
import org.springframework.web.multipart.MultipartFile;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

import static org.mockito.Mockito.*;
import static org.junit.jupiter.api.Assertions.*;

class PostControllerTest {

    @InjectMocks
    private PostController postController;

    @Mock
    private PostService postService;

    @BeforeEach
    void setup() {
        MockitoAnnotations.openMocks(this);
    }

    // 测试创建帖子
    void createPostTest() {
        Long userId = 1L;
        String postContent = "Test Post Content";
        String title = "Test Post Title";
        MultipartFile[] imageFiles = new MultipartFile[0];  // Empty array for simplicity

        // 模拟服务层的返回值
        Map<String, Object> responseMap = new HashMap<>();
        responseMap.put("postNo", 123L);
        responseMap.put("message", "帖子创建成功");

        when(postService.createPost(userId, postContent, title, imageFiles)).thenReturn(responseMap);

        // 调用控制器的方法
        Map<String, Object> resultMap = postController.createPost(userId, postContent, title, imageFiles);

        // 手动将 Map<String, Object> 包装到 Result.success()
        Result<Map<String, Object>> result = Result.success(resultMap, "帖子创建成功");

        // 验证返回的结果
        assertEquals("200", result.getCode());
        assertEquals("帖子创建成功", result.getMsg());
        assertEquals(123L, result.getData().get("postNo"));
    }

    // 测试编辑帖子
    @Test
    void updatePostTest() {
        Long postId = 1L;
        Post post = new Post();
        post.setPostNo(postId);
        post.setTitle("Updated Title");
        post.setPostContent("Updated Content");

        // 模拟服务层的行为
        doNothing().when(postService).updatePost(postId, post);

        // 调用控制器的方法
        String result = postController.updatePost(postId, post);

        // 验证返回的结果
        assertEquals("Post updated successfully!", result);
    }

    // 测试删除帖子
    @Test
    void deletePostTest() {
        Long postId = 1L;

        // 模拟服务层的行为
        doNothing().when(postService).deletePost(postId);

        // 调用控制器的方法
        String result = postController.deletePost(postId);

        // 验证返回的结果
        assertEquals("Post deleted successfully!", result);
    }

    // 测试点赞帖子
    @Test
    void likePostTest() {
        Long postId = 1L;
        Long userId = 1L;

        // 模拟服务层的行为
        doNothing().when(postService).likePost(postId, userId);

        // 创建请求体
        Map<String, Long> requestBody = new HashMap<>();
        requestBody.put("user_id", userId);

        // 调用控制器的方法
        String result = postController.likePost(postId, requestBody);

        // 验证返回的结果
        assertEquals("Post liked successfully!", result);
    }

    // 测试取消点赞帖子
    @Test
    void unlikePostTest() {
        Long postId = 1L;
        Long userId = 1L;

        // 模拟服务层的行为
        doNothing().when(postService).unlikePost(postId, userId);

        // 创建请求体
        Map<String, Long> requestBody = new HashMap<>();
        requestBody.put("user_id", userId);

        // 调用控制器的方法
        String result = postController.unlikePost(postId, requestBody);

        // 验证返回的结果
        assertEquals("Post unliked successfully!", result);
    }

    // 测试收藏帖子
    @Test
    void collectPostTest() {
        Long postId = 1L;
        Long userId = 1L;

        // 模拟服务层的行为
        doNothing().when(postService).collectPost(postId, userId);

        // 创建请求体
        Map<String, Long> requestBody = new HashMap<>();
        requestBody.put("user_id", userId);

        // 调用控制器的方法
        String result = postController.collectPost(postId, requestBody);

        // 验证返回的结果
        assertEquals("Post collected successfully!", result);
    }

    // 测试取消收藏帖子
    @Test
    void uncollectPostTest() {
        Long postId = 1L;
        Long userId = 1L;

        // 模拟服务层的行为
        doNothing().when(postService).uncollectPost(postId, userId);

        // 创建请求体
        Map<String, Long> requestBody = new HashMap<>();
        requestBody.put("user_id", userId);

        // 调用控制器的方法
        String result = postController.uncollectPost(postId, requestBody);

        // 验证返回的结果
        assertEquals("Post uncollected successfully!", result);
    }

    // 测试获取所有帖子
    @Test
    void getAllPostsTest() {
        // 模拟服务层的行为
        Map<String, Object> responseMap = new HashMap<>();
        responseMap.put("total", 5);
        responseMap.put("posts", new ArrayList<>());

        when(postService.getAllPosts()).thenReturn(responseMap);

        // 调用控制器的方法
        Map<String, Object> result = postController.getAllPosts();

        // 验证返回的结果
        assertEquals(5, result.get("total"));
    }

    // 测试根据帖子ID获取帖子
    @Test
    void getPostByIdTest() {
        Long postId = 1L;
        Map<String, Object> responseMap = new HashMap<>();
        responseMap.put("postNo", 123L);
        responseMap.put("message", "Post details");

        // 模拟服务层的行为
        when(postService.getPostById(postId)).thenReturn(responseMap);

        // 调用控制器的方法
        Map<String, Object> result = postController.getPostById(postId);

        // 验证返回的结果
        assertEquals("Post details", result.get("message"));
        assertEquals(123L, result.get("postNo"));
    }
}
