package com.example.myproject.controller;

import com.example.myproject.controller.TaskController;
import com.example.myproject.service.TaskService;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.springframework.http.ResponseEntity;

import java.util.HashMap;
import java.util.Map;

import static org.mockito.Mockito.*;
import static org.junit.jupiter.api.Assertions.*;

class TaskControllerTest {

    @InjectMocks
    private TaskController taskController;

    @Mock
    private TaskService taskService;

    @BeforeEach
    void setup() {
        MockitoAnnotations.openMocks(this);
    }

    // 测试获取当前用户的新手任务列表接口
    @Test
    void testGetAllTasks() {
        Long userId = 1L;
        Map<String, Object> mockResponse = new HashMap<>();
        mockResponse.put("tasks", "mock_tasks");

        // 模拟 taskService.getAllTasksForUser 返回
        when(taskService.getAllTasksForUser(userId)).thenReturn(mockResponse);

        // 创建请求体
        Map<String, Object> request = new HashMap<>();
        request.put("user_id", userId);

        // 调用控制器方法
        Map<String, Object> response = taskController.getAllTasks(request);

        // 验证返回的结果
        assertEquals("mock_tasks", response.get("tasks"));
    }

    // 测试更新任务状态接口
    @Test
    void testUpdateTaskStatus() {
        Long userId = 1L;
        String taskId = "task_123";
        Map<String, Object> mockResponse = new HashMap<>();
        mockResponse.put("status", "success");
        mockResponse.put("message", "任务状态已更新");

        // 模拟 taskService.updateTaskStatus 返回
        when(taskService.updateTaskStatus(userId, taskId)).thenReturn(mockResponse);

        // 创建请求体
        Map<String, Object> request = new HashMap<>();
        request.put("user_id", userId);
        request.put("task_id", taskId);

        // 调用控制器方法
        Map<String, Object> response = taskController.updateTaskStatus(request);

        // 验证返回的结果
        assertEquals("success", response.get("status"));
        assertEquals("任务状态已更新", response.get("message"));
    }

    // 测试获取当前经验和任务奖励接口
    @Test
    void testGetExperience() {
        Long userId = 1L;
        Map<String, Object> mockResponse = new HashMap<>();
        mockResponse.put("current_experience", 1500);
        mockResponse.put("level", "Intermediate");
        mockResponse.put("reward", Map.of("experience", 1000, "points", 200));

        // 模拟 taskService.getUserExperience 返回
        when(taskService.getUserExperience(userId)).thenReturn(mockResponse);

        // 创建请求体
        Map<String, Object> request = new HashMap<>();
        request.put("user_id", userId);

        // 调用控制器方法
        Map<String, Object> response = taskController.getExperience(request);

        // 验证返回的结果
        assertEquals(1500, response.get("current_experience"));
        assertEquals("Intermediate", response.get("level"));
        assertTrue(response.containsKey("reward"));
    }

    // 测试获取当前的指引步骤接口
    @Test
    void testGetNewStep() {
        Long userId = 1L;
        Map<String, Object> mockResponse = new HashMap<>();
        mockResponse.put("current_step", "step_1");
        mockResponse.put("total_steps", 5);
        mockResponse.put("step_description", "Complete the introduction task");

        // 模拟 taskService.getNewStep 返回
        when(taskService.getNewStep(userId)).thenReturn(mockResponse);

        // 创建请求体
        Map<String, Object> request = new HashMap<>();
        request.put("user_id", userId);

        // 调用控制器方法
        Map<String, Object> response = taskController.getNewStep(request);

        // 验证返回的结果
        assertEquals("step_1", response.get("current_step"));
        assertEquals(5, response.get("total_steps"));
        assertEquals("Complete the introduction task", response.get("step_description"));
    }

    // 测试更新进度接口
    @Test
    void testUpdateProgress() {
        Long userId = 1L;
        String taskId = "task_123";
        Integer progress = 50;
        Map<String, Object> mockResponse = new HashMap<>();
        mockResponse.put("status", "success");
        mockResponse.put("message", "进度已更新");

        // 模拟 taskService.updateTaskProgress 返回
        when(taskService.updateTaskProgress(userId, taskId, progress)).thenReturn(mockResponse);

        // 创建请求体
        Map<String, Object> request = new HashMap<>();
        request.put("user_id", userId);
        request.put("task_id", taskId);
        request.put("progress", progress);

        // 调用控制器方法
        Map<String, Object> response = taskController.updateProgress(request);

        // 验证返回的结果
        assertEquals("success", response.get("status"));
        assertEquals("进度已更新", response.get("message"));
    }

    // 测试领取任务奖励接口
    @Test
    void testRewardClaim() {
        Long userId = 1L;
        String taskId = "task_123";
        Map<String, Object> mockResponse = new HashMap<>();
        mockResponse.put("status", "success");
        mockResponse.put("message", "奖励已领取");
        mockResponse.put("reward", Map.of("experience", 1000, "points", 200));

        // 模拟 taskService.claimReward 返回
        when(taskService.claimReward(userId, taskId)).thenReturn(mockResponse);

        // 创建请求体
        Map<String, Object> request = new HashMap<>();
        request.put("user_id", userId);
        request.put("task_id", taskId);

        // 调用控制器方法
        Map<String, Object> response = taskController.rewardClaim(request);

        // 验证返回的结果
        assertEquals("success", response.get("status"));
        assertEquals("奖励已领取", response.get("message"));
        assertTrue(response.containsKey("reward"));
    }

    // 测试检查任务奖励状态接口
    @Test
    void testRewardReview() {
        Long userId = 1L;
        String taskId = "task_123";
        Map<String, Object> mockResponse = new HashMap<>();
        mockResponse.put("status", "reward_not_issued");
        mockResponse.put("message", "任务奖励未被领取");

        // 模拟 taskService.checkRewardStatus 返回
        when(taskService.checkRewardStatus(userId, taskId)).thenReturn(mockResponse);

        // 创建请求体
        Map<String, Object> request = new HashMap<>();
        request.put("user_id", userId);
        request.put("task_id", taskId);

        // 调用控制器方法
        Map<String, Object> response = taskController.rewardReview(request);

        // 验证返回的结果
        assertEquals("reward_not_issued", response.get("status"));
        assertEquals("任务奖励未被领取", response.get("message"));
    }
}
