package com.example.myproject.controller;

import com.example.myproject.entity.Users;
import com.example.myproject.service.UserService;
import com.example.myproject.repository.UserRepository;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import static org.mockito.Mockito.*;
import static org.junit.jupiter.api.Assertions.*;

class UserControllerTest {

    @Mock
    private UserService userService;

    @Mock
    private UserRepository userRepository;

    @InjectMocks
    private UserController userController;

    @BeforeEach
    void setUp() {
        // 初始化模拟对象
        MockitoAnnotations.openMocks(this);
    }

    // 测试生成邀请码接口
    @Test
    void testGenerateInviteCode() {
        // 设置输入数据
        Map<String, Object> request = new HashMap<>();
        request.put("user_id", 1L);

        // 模拟服务层的返回
        Map<String, Object> responseMap = new HashMap<>();
        responseMap.put("status", "success");
        responseMap.put("message", "邀请码生成成功");
        responseMap.put("invite_code", "ABC123XYZ");

        when(userService.generateInviteCode(1L)).thenReturn(responseMap);

        // 调用控制器方法
        Map<String, Object> resultMap = userController.generateInviteCode(request);

        // 验证返回结果
        assertEquals("success", resultMap.get("status"));
        assertEquals("邀请码生成成功", resultMap.get("message"));
        assertEquals("ABC123XYZ", resultMap.get("invite_code"));

        // 验证服务层方法是否被调用
        verify(userService, times(1)).generateInviteCode(1L);
    }

    // 测试用户修改密码接口
    @Test
    void testChangePassword() {
        Long userId = 1L;
        Map<String, Object> request = new HashMap<>();
        request.put("user_id", userId);
        request.put("old_password", "oldPassword");
        request.put("new_password", "newPassword");
        request.put("confirm_password", "newPassword");

        // 模拟服务层的返回
        String resultMessage = "密码修改成功";
        when(userService.changePassword(userId, "oldPassword", "newPassword", "newPassword"))
                .thenReturn(resultMessage);

        // 调用控制器方法
        Map<String, Object> resultMap = userController.changePassword(request);

        // 验证返回结果
        assertEquals("密码修改成功", resultMap.get("message"));
        assertEquals("success", resultMap.get("status"));

        // 验证服务层方法是否被调用
        verify(userService, times(1)).changePassword(userId, "oldPassword", "newPassword", "newPassword");
    }

    // 测试用户注册接口
    @Test
    void testRegister() {
        // 设置输入数据
        Map<String, Object> request = new HashMap<>();
        request.put("username", "testUser");
        request.put("email", "test@example.com");
        request.put("password", "password");
        request.put("role", "USER");
        request.put("inviteCode", "12345");

        // 模拟服务层的返回
        String resultMessage = "用户注册成功";
        when(userService.registerUser("testUser", "test@example.com", "password", "USER", "12345"))
                .thenReturn(resultMessage);

        // 调用控制器方法
        Map<String, Object> resultMap = userController.register(request);

        // 验证返回结果
        assertEquals("用户注册成功", resultMap.get("msg"));

        // 验证服务层方法是否被调用
        verify(userService, times(1)).registerUser("testUser", "test@example.com", "password", "USER", "12345");
    }

    @Test
    void testLogin() {
        String username = "testUser";
        String password = "password";

        // 模拟服务层的返回
        String loginMessage = "登录成功";
        when(userService.loginUser("testUser", "password")).thenReturn(loginMessage);

        // 模拟用户查询
        Users user = new Users();
        user.setUserId(1L);
        user.setUsername("testUser");
        user.setEmail("test@example.com");
        when(userRepository.findByUsername("testUser")).thenReturn(Optional.of(user));

        // 设置请求数据
        Map<String, Object> request = new HashMap<>();
        request.put("username", username);
        request.put("password", password);

        // 调用控制器方法
        Map<String, Object> resultMap = userController.login(request);

        // 验证返回结果
        assertEquals("登录成功", resultMap.get("msg"));

        // 验证 user 不是 Map 类型，而是 Users 对象
        Users loggedInUser = (Users) resultMap.get("user");
        assertNotNull(loggedInUser);
        assertEquals("testUser", loggedInUser.getUsername());
        assertEquals("test@example.com", loggedInUser.getEmail());

        // 验证服务层方法是否被调用
        verify(userService, times(1)).loginUser("testUser", "password");
    }


    // 测试获取用户个人资料接口
    @Test
    void testGetProfile() {
        Long userId = 1L;

        // Mock 用户数据
        Users user = new Users();
        user.setUserId(userId);
        user.setUsername("testUser");
        user.setEmail("test@example.com");
        user.setAvatarUrl("https://example.com/avatar.jpg");

        // 模拟服务层返回
        Map<String, Object> profileData = new HashMap<>();
        profileData.put("avatarUrl", user.getAvatarUrl());
        profileData.put("username", user.getUsername());
        profileData.put("email", user.getEmail());
        when(userService.getProfile(userId)).thenReturn(profileData);

        // 调用控制器方法
        Map<String, Object> resultMap = userController.getProfile(userId);

        // 验证返回结果
        assertEquals("testUser", resultMap.get("username"));
        assertEquals("test@example.com", resultMap.get("email"));
        assertEquals("https://example.com/avatar.jpg", resultMap.get("avatarUrl"));

        // 验证服务层方法是否被调用
        verify(userService, times(1)).getProfile(userId);
    }

    // 测试修改用户个人资料接口
    @Test
    void testEditProfile() {
        Long userId = 1L;
        Map<String, Object> profileData = new HashMap<>();
        profileData.put("avatarUrl", "https://example.com/avatar.jpg");
        profileData.put("nickname", "newNickname");
        profileData.put("gender", "Male");
        profileData.put("description", "Updated description");
        profileData.put("hobbies", "Reading, Hiking");

        // 模拟服务层的返回
        when(userService.editProfile(userId, "https://example.com/avatar.jpg", "newNickname", "Male", "Updated description", "Reading, Hiking"))
                .thenReturn(true);

        // 调用控制器方法
        Map<String, String> resultMap = userController.editProfile(userId, profileData);

        // 验证返回结果
        assertEquals("用户资料更新成功", resultMap.get("message"));

        // 验证服务层方法是否被调用
        verify(userService, times(1)).editProfile(userId, "https://example.com/avatar.jpg", "newNickname", "Male", "Updated description", "Reading, Hiking");
    }

    @Test
    public void testCalculateShareRate() {
        Long userId = 1L;

        // 模拟 UserService 返回的结果
        Map<String, Object> expectedResponse = Map.of(
                "status", "success",
                "message", "分享率计算成功",
                "shareRate", 0.5f
        );

        // 设置用户数据
        Users user = new Users();
        user.setUserId(userId);
        user.setUploadCount(50f);  // 上传量为50
        user.setDownloadCount(100f);  // 下载量为100
        when(userService.calculateShareRate(userId)).thenReturn(expectedResponse);

        // 调用接口
        Map<String, Object> result = userController.calculateShareRate(userId);

        // 验证结果
        assertEquals(expectedResponse, result);
        verify(userService, times(1)).calculateShareRate(userId);  // 验证服务方法是否被调用
    }
}
