package com.example.myproject.controller;

import cn.dev33.satoken.annotation.SaCheckLogin;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.example.myproject.common.base.PageUtil;
import com.example.myproject.dto.param.TorrentParam;
import com.example.myproject.dto.vo.TorrentVO;
import com.example.myproject.entity.TorrentEntity;
import com.example.myproject.mapper.UserMapper;
import com.example.myproject.mapper.VerificationTokenMapper;
import com.example.myproject.entity.User;
import com.example.myproject.entity.VerificationToken;
import com.example.myproject.service.EmailService;
import com.example.myproject.service.UserService;
import com.example.myproject.common.base.Result;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import org.apache.commons.lang3.RandomStringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.List;

@RestController
@RequestMapping("/user")
@Api(value = "用户管理接口", tags = {"用户管理"})
public class UserController {

    @Resource
    private UserService userService;

    @Autowired
    private AuthenticationManager authenticationManager;

    @Autowired
    private UserMapper userMapper; // 使用 MyBatis-Plus

    @Autowired
    private VerificationTokenMapper verificationTokenMapper; // 替换 JPA

    private static final Logger logger = LoggerFactory.getLogger(UserController.class);

    @PostMapping("/login")
    @ApiOperation(value = "用户登录", notes = "使用用户名和密码进行登录")
    public Result loginController(@RequestParam @ApiParam(value = "用户名", required = true) String username,
                                  @RequestParam @ApiParam(value = "密码", required = true) String password) {
        try {
            Authentication authentication = authenticationManager.authenticate(
                    new UsernamePasswordAuthenticationToken(username, password)
            );
            SecurityContextHolder.getContext().setAuthentication(authentication);

            // 使用 MyBatis-Plus 查询
            User user = userMapper.selectOne(new QueryWrapper<User>().eq("username", username));

            System.out.println("Login successful for user: " + username);
            return Result.ok(user);
        } catch (AuthenticationException e) {
            return Result.error("登录失败");
        }
    }

    @PostMapping("/register")
    @ApiOperation(value = "用户注册", notes = "使用用户信息进行注册")
    public Result registerController(@RequestBody @ApiParam(value = "新用户信息", required = true) User newUser) {
        if (userService.checkEmailExists(newUser.getEmail())) {
            return Result.error( "邮箱已被使用，请使用其他邮箱注册或找回密码！");
        }
        boolean success = userService.preRegisterUser(newUser);
        if (success) {
            User responseUser = new User();
            responseUser.setEmail(newUser.getEmail());
            return Result.ok();
        } else {
            return Result.error("账号已存在或注册失败！");
        }
    }

    public static class VerificationRequest {
        private String email;
        private String code;

        public String getEmail() { return email; }
        public void setEmail(String email) { this.email = email; }
        public String getCode() { return code; }
        public void setCode(String code) { this.code = code; }
    }

    @PostMapping("/verify-code")
    @ApiOperation(value = "验证邮箱验证码", notes = "验证用户邮箱的验证码")
    public Result verifyEmailCode(@RequestBody @ApiParam(value = "验证请求信息", required = true) VerificationRequest verificationRequest) {
        String email = verificationRequest.getEmail();
        String code = verificationRequest.getCode();
        boolean isVerified = userService.verifyEmail(email, code);
        if (isVerified) {
            return Result.ok();
        } else {
            return Result.error( "验证码错误或已过期！");
        }
    }

    @Autowired
    private EmailService emailService;

    public static class EmailRequest {
        private String email;
        public String getEmail() { return email; }
        public void setEmail(String email) { this.email = email; }
    }

    @PostMapping("/get-verification-email")
    @ApiOperation(value = "发送验证邮件", notes = "通过电子邮件发送验证邮件")
    public ResponseEntity<Result> sendVerificationEmail(@RequestBody @ApiParam(value = "发送验证请求", required = true) EmailRequest emailVerificationRequest) {
        String email = emailVerificationRequest.getEmail();
        User user = userMapper.selectOne(new QueryWrapper<User>().eq("email", email));
        if (user == null) {
            logger.error("未找到与该邮箱地址相关联的用户: {}", email);
            return ResponseEntity.status(HttpStatus.BAD_REQUEST)
                    .body(Result.error("未找到与该邮箱地址相关联的用户"));
        }

        // 生成验证码
        String token = RandomStringUtils.randomNumeric(6);
        Instant expiryDate = Instant.now().plus(1, ChronoUnit.HOURS);
        logger.info("生成的验证令牌: {}, 过期时间: {}", token, expiryDate);

        VerificationToken verificationToken = new VerificationToken(token, user.getUsername(), email, user.getPassword(), expiryDate);

        // 保存到 MyBatis-Plus 数据库
        verificationTokenMapper.insert(verificationToken);

        logger.info("验证令牌已保存，用户: {}", user.getUsername());
        emailService.sendVerificationEmail(email, token);

        return ResponseEntity.ok(Result.ok());
    }
    @PostMapping("/checkPassword")
    public Result<String> checkPassword(@RequestParam Long userId, @RequestParam String password) {
        boolean isPasswordCorrect = userService.checkPassword(userId, password);
        if (isPasswordCorrect) {
            return Result.ok();
        } else {
            return Result.error("原始密码输入错误");
        }
    }


//    @SaCheckLogin
//    @Operation(summary = "用户收藏列表", description = "获取用户收藏的种子列表-分页-排序")
//    @ApiResponse(responseCode = "0", description = "操作成功",
//            content = {@Content(mediaType = "application/json",
//                    schema = @Schema(implementation = TorrentVO.class))
//            })
//    @PostMapping("/favorite/list")
//    public Result listFavorites(@RequestBody FavoriteParam param) {
//        if (param.getUserId() == null) {
//            return Result.error("缺少 userId");
//        }
//
//        // 校验排序字段是否合理（可选）
//        param.validOrder(param.getOrderKey(TorrentEntity.class));
//
//        PageUtil.startPage(param);
//
//        List<TorrentEntity> list = favoriteService.getUserFavoritesPaged(param.getUserId());
//
//        return Result.ok(list, PageUtil.getPage(list));
//    }
//



}
