package com.example.myproject.service.serviceImpl;

import com.example.myproject.entity.Promotion;
import com.example.myproject.mapper.PromotionMapper;
import com.example.myproject.service.PromotionService;
import com.example.myproject.dto.PromotionCreateDTO;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.List;
import java.util.stream.Collectors;

@Service
public class PromotionServiceImpl implements PromotionService {

    @Autowired
    private PromotionMapper promotionMapper;

    @Override
    @Transactional
    public Promotion createPromotion(PromotionCreateDTO promotionDTO) {
        // 验证时间
        LocalDateTime now = LocalDateTime.now();
        if (promotionDTO.getEndTime().isBefore(now)) {
            throw new RuntimeException("结束时间不能早于当前时间");
        }

        // 验证种子ID是否存在
        validateTorrentIds(promotionDTO.getApplicableTorrentIds());

        // 创建促销活动
        Promotion promotion = new Promotion();
        promotion.setName(promotionDTO.getName());
        promotion.setDescription(promotionDTO.getDescription());
        promotion.setStartTime(promotionDTO.getStartTime());
        promotion.setEndTime(promotionDTO.getEndTime());
        promotion.setDiscountPercentage(promotionDTO.getDiscountPercentage());

        // 把List<Long>转换成逗号分隔字符串
        String applicableTorrentIdsStr = promotionDTO.getApplicableTorrentIds().stream()
                .map(String::valueOf)
                .collect(Collectors.joining(","));
        promotion.setApplicableTorrentIds(applicableTorrentIdsStr);

        promotion.setCreateTime(now);
        promotion.setUpdateTime(now);
        promotion.setIsDeleted(false);

        promotionMapper.insert(promotion);
        return promotion;
    }

    @Override
    public List<Promotion> getAllActivePromotions() {
        LocalDateTime now = LocalDateTime.now();
        return promotionMapper.findActivePromotions(now);
    }

    @Override
    public Promotion getPromotionById(Long promotionId) {
        Promotion promotion = promotionMapper.selectById(promotionId);
        if (promotion == null || promotion.getIsDeleted()) {
            return null;
        }
        return promotion;
    }

    @Override
    @Transactional
    public void deletePromotion(Long promotionId) {
        Promotion promotion = getPromotionById(promotionId);
        if (promotion == null) {
            throw new RuntimeException("促销活动不存在");
        }

        // 软删除
        promotion.setIsDeleted(true);
        promotion.setUpdateTime(LocalDateTime.now());
        promotionMapper.updateById(promotion);
    }

    @Override
    public double getCurrentDiscount(Long torrentId) {
        LocalDateTime now = LocalDateTime.now();
        List<Promotion> activePromotions = promotionMapper.findActivePromotionsForTorrent(torrentId, now);

        // 如果有多个促销活动，取折扣最大的
        return activePromotions.stream()
                .mapToDouble(Promotion::getDiscountPercentage)
                .max()
                .orElse(0.0);
    }

    /**
     * 验证种子ID是否存在
     */
    private void validateTorrentIds(List<Long> torrentIds) {
        if (torrentIds == null || torrentIds.isEmpty()) {
            throw new RuntimeException("适用种子列表不能为空");
        }

        // 检查所有种子ID是否都存在
        List<Long> invalidIds = torrentIds.stream()
                .filter(id -> promotionMapper.checkTorrentExists(id) == 0)  // 改成 == 0
                .collect(Collectors.toList());

        if (!invalidIds.isEmpty()) {
            throw new RuntimeException("以下种子ID不存在: " + invalidIds);
        }
    }
}
