
package com.example.myproject.service.serviceImpl;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.example.myproject.mapper.UserMapper;
import com.example.myproject.mapper.VerificationTokenMapper;
import com.example.myproject.entity.User;
import com.example.myproject.entity.VerificationToken;
import com.example.myproject.service.EmailService;
import com.example.myproject.service.UserService;
import org.apache.commons.lang3.RandomStringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.stereotype.Service;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
//登录注册
@Service
public class UserServiceImpl extends ServiceImpl<UserMapper, User> implements UserService {
    private static final Logger log = LoggerFactory.getLogger(UserServiceImpl.class);
    @Autowired
    private UserMapper userMapper; // Using MyBatis-Plus mapper
    @Autowired
    private EmailServiceImpl emailService;
    @Autowired
    private VerificationTokenMapper verificationTokenMapper; // Using MyBatis-Plus mapper
    @Autowired
    private PasswordEncoder passwordEncoder; // Injecting password encoder
    @Override
    public User loginService(String username, String password) {
        log.debug("Attempting login with username: {}, password: {}", username, password);
        User user = userMapper.selectOne(new QueryWrapper<User>().eq("username", username).eq("password", password));
        if (user != null) {
            user.setPassword("");  // Clear the password before returning
            log.debug("Login successful, User ID: {}", user.getId());
        } else {
            log.debug("Login failed, incorrect username or password.");
        }
        return user;
    }
    @Override
    public boolean preRegisterUser(User user) {
        log.debug("Pre-registering user, username: {}, email: {}", user.getUsername(), user.getEmail());
        // 检查用户名或邮箱是否已存在
        boolean userExists = userMapper.selectOne(new QueryWrapper<User>().eq("username", user.getUsername())) != null ||
                userMapper.selectOne(new QueryWrapper<User>().eq("email", user.getEmail())) != null;
        if (userExists) {
            log.debug("Pre-registration failed, username or email already exists.");
            return false;  // 用户名或邮箱已经存在
        }
        // 加密密码
        String encryptedPassword = passwordEncoder.encode(user.getPassword());
        // 生成验证码
        String token = RandomStringUtils.randomNumeric(6);
        // 设置过期时间为当前时间加一小时
        Instant expiryDate = Instant.now().plus(1, ChronoUnit.HOURS);
        // 创建验证令牌对象
        VerificationToken verificationToken = new VerificationToken(
                token,
                user.getUsername(),
                user.getEmail(),
                encryptedPassword,
                expiryDate
        );
        // 插入验证令牌
        int rowsInserted = verificationTokenMapper.insert(verificationToken);
        if (rowsInserted > 0) {
            // 发送验证邮件
            emailService.sendVerificationEmail(user.getEmail(), token);
            log.debug("Pre-registration successful, verification code: {}, expiry date: {}", token, expiryDate);
            return true;  // 注册前验证成功
        } else {
            log.error("Failed to insert verification token into database.");
            return false;  // 如果插入验证令牌失败，返回失败
        }
    }
    @Override
    public boolean verifyEmail(String email, String token) {
        log.debug("Verifying email, email: {}, token: {}", email, token);
        VerificationToken verificationToken = verificationTokenMapper.selectOne(
                new QueryWrapper<VerificationToken>().eq("token", token).eq("email", email)
        );
        if (verificationToken != null && !verificationToken.isExpired()) {
            log.debug("Verification code is valid, username: {}", verificationToken.getUsername());
            User user = userMapper.selectOne(new QueryWrapper<User>().eq("email", email));
            if (user == null) {
                user = new User();
                user.setEmail(email);
                user.setUsername(verificationToken.getUsername());
                user.setPassword(verificationToken.getPassword());
                user.setEmailVerified(true);
                userMapper.insert(user);  // Save new user
                log.debug("New user created, User ID: {}", user.getId());
            } else {
                user.setEmailVerified(true);
                userMapper.updateById(user);  // Update existing user
                log.debug("User email verified, User ID: {}", user.getId());
            }
            verificationTokenMapper.delete(new QueryWrapper<VerificationToken>().eq("token", token)); // Clean up the token
            log.debug("Verification code deleted.");
            return true;
        }
        log.debug("Verification code is invalid or expired.");
        return false;
    }
    public boolean checkEmailExists(String email) {
        log.debug("检查邮箱是否存在，邮箱：{}", email);
        boolean exists = userMapper.selectCount(new QueryWrapper<User>().eq("email", email)) > 0;
        log.debug("邮箱存在状态：{}", exists);
        return exists;
    }
    public boolean checkPassword(Long userId, String password) {
        User user = userMapper.selectById(userId);
        if (user == null) {
            throw new RuntimeException("用户不存在");
        }
        String encryptedPassword = user.getPassword();
        return passwordEncoder.matches(password, encryptedPassword);
    }
}
