package com.example.myproject.controller;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.example.myproject.entity.User;
import com.example.myproject.service.EmailService;
import com.example.myproject.service.UserService;
import com.example.myproject.mapper.UserMapper;
import com.example.myproject.mapper.VerificationTokenMapper;
import com.example.myproject.common.base.Result;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.*;
import org.springframework.http.ResponseEntity;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.Mockito.*;

class UserControllerTest {

    @InjectMocks
    private UserController userController;

    @Mock
    private UserService userService;
    @Mock
    private UserMapper userMapper;
    @Mock
    private EmailService emailService;
    @Mock
    private AuthenticationManager authenticationManager;
    @Mock
    private VerificationTokenMapper verificationTokenMapper;

    @BeforeEach
    void setup() {
        MockitoAnnotations.openMocks(this);
    }

    @Test
    void loginController_success() {
        String username = "testuser";
        String password = "testpass";
        User mockUser = new User();
        mockUser.setUsername(username);

        Authentication mockAuth = mock(Authentication.class);
        when(authenticationManager.authenticate(any())).thenReturn(mockAuth);
        when(userMapper.selectOne(any(QueryWrapper.class))).thenReturn(mockUser);

        Result result = userController.loginController(username, password);

        assertEquals("200", result.getCode());
        assertEquals(mockUser, result.getData());
    }


    @Test
    void loginController_failure() {
        String username = "user";
        String password = "wrongpass";

        // 模拟认证失败，抛出 Bad credentials 异常
        when(authenticationManager.authenticate(any()))
                .thenThrow(new AuthenticationException("Bad credentials") {});


        // 调用登录接口
        Result result = userController.loginController(username, password);

        // 断言返回的状态码和消息
        assertEquals("401", result.getCode());
        assertTrue(result.getMsg().contains("登录失败"));
    }

    @Test
    void registerController_emailExists() {
        User user = new User();
        user.setEmail("test@example.com");

        when(userService.checkEmailExists(user.getEmail())).thenReturn(true);

        Result result = userController.registerController(user);

        assertEquals("邮箱冲突", result.getCode());
    }

    @Test
    void registerController_success() {
        User user = new User();
        user.setEmail("test@example.com");

        when(userService.checkEmailExists(user.getEmail())).thenReturn(false);
        when(userService.preRegisterUser(user)).thenReturn(true);

        Result result = userController.registerController(user);

        assertEquals("200", result.getCode());
        assertEquals(user.getEmail(), ((User) result.getData()).getEmail());
    }

    @Test
    void verifyEmailCode_success() {
        when(userService.verifyEmail("test@example.com", "123456")).thenReturn(true);

        UserController.VerificationRequest request = new UserController.VerificationRequest();
        request.setEmail("test@example.com");
        request.setCode("123456");

        Result result = userController.verifyEmailCode(request);

        assertEquals("200", result.getCode());
    }

    @Test
    void verifyEmailCode_failure() {
        when(userService.verifyEmail("test@example.com", "000000")).thenReturn(false);

        UserController.VerificationRequest request = new UserController.VerificationRequest();
        request.setEmail("test@example.com");
        request.setCode("000000");

        Result result = userController.verifyEmailCode(request);

        assertEquals("验证失败", result.getCode());
    }

    @Test
    void checkPassword_success() {
        when(userService.checkPassword(1L, "abc123")).thenReturn(true);

        Result<String> result = userController.checkPassword(1L, "abc123");

        assertEquals("200", result.getCode());
        assertEquals("原始密码输入正确", result.getMsg());
    }

    @Test
    void checkPassword_failure() {
        when(userService.checkPassword(1L, "wrong")).thenReturn(false);

        Result<String> result = userController.checkPassword(1L, "wrong");

        assertEquals("305", result.getCode());
        assertEquals("原始密码输入错误", result.getMsg());
    }

    @Test
    void sendVerificationEmail_userNotFound() {
        UserController.EmailRequest request = new UserController.EmailRequest();
        request.setEmail("notfound@example.com");

        when(userMapper.selectOne(any())).thenReturn(null);

        ResponseEntity<Result> response = userController.sendVerificationEmail(request);

        assertEquals(400, response.getStatusCodeValue());
        assertEquals("1", response.getBody().getCode());
    }
}
