import React, { useState } from 'react';
import axios from 'axios';
import Header from '../../components/Header'; // 导入 Header 组件

const API_BASE = process.env.REACT_APP_API_BASE;

const PublishSeed = () => {
  const [title, setTitle] = useState('');
  const [description, setDescription] = useState('');
  const [tags, setTags] = useState([]);
  const [category, setCategory] = useState('movie');
  const [file, setFile] = useState(null);
  const [imageUrl, setImageUrl] = useState('');
  const [message, setMessage] = useState('');
  const [isLoading, setIsLoading] = useState(false);

  const handleTagsChange = (e) => {
    setTags(e.target.value.split(',').map(tag => tag.trim()));
  };

  const handleFileChange = (e) => {
    setFile(e.target.files[0]);
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    setIsLoading(true);
    setMessage('');

    const formData = new FormData();
    formData.append('title', title);
    formData.append('description', description);
    formData.append('tags', JSON.stringify(tags)); // Tags as JSON array
    formData.append('category', category);
    formData.append('file', file);
    formData.append('image_url', imageUrl);

    try {
      const response = await axios.post(`${API_BASE}/echo/seeds/upload`, formData, {
        headers: {
          'Content-Type': 'multipart/form-data',
        },
      });

      if (response.data.status === 'success') {
        setMessage('种子上传成功');
      } else {
        setMessage('上传失败，请稍后再试');
      }
    } catch (error) {
      console.error(error);
      setMessage('上传失败，发生了错误');
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <div className="publish-seed-container">
      <Header /> {/* 在这里插入导航栏 */}
      {message && <div className="message">{message}</div>}
      <form onSubmit={handleSubmit} encType="multipart/form-data">
        <div className="form-group">
          <label>标题</label>
          <input
            type="text"
            value={title}
            onChange={(e) => setTitle(e.target.value)}
            required
          />
        </div>

        <div className="form-group">
          <label>描述</label>
          <textarea
            value={description}
            onChange={(e) => setDescription(e.target.value)}
            required
          />
        </div>

        <div className="form-group">
          <label>标签 (逗号分隔)</label>
          <input
            type="text"
            value={tags.join(', ')}
            onChange={handleTagsChange}
            placeholder="例如：科幻, 动作"
            required
          />
        </div>

        <div className="form-group">
          <label>分类</label>
          <select
            value={category}
            onChange={(e) => setCategory(e.target.value)}
            required
          >
            <option value="movie">电影</option>
            <option value="tv">电视剧</option>
            <option value="music">音乐</option>
          </select>
        </div>

        <div className="form-group">
          <label>种子文件</label>
          <input
            type="file"
            onChange={handleFileChange}
            required
          />
        </div>

        <div className="form-group">
          <label>封面图URL</label>
          <input
            type="url"
            value={imageUrl}
            onChange={(e) => setImageUrl(e.target.value)}
            placeholder="例如：http://example.com/images/cover.jpg"
            required
          />
        </div>

        <div className="form-group">
          <button type="submit" disabled={isLoading}>
            {isLoading ? '正在上传...' : '上传种子'}
          </button>
        </div>
      </form>
    </div>
  );
};

export default PublishSeed;
