import React, { useEffect, useState } from 'react';
import axios from 'axios';
import { Link } from 'wouter';
import { GoodTwo, Star, Delete } from '@icon-park/react';
import { likePost } from '../../posts-detail/api';
import { formatAvatarUrl } from '../../../../components/utils/avatar';
import './PostList.css';

// 修改后的封面图 URL 拼接函数
const formatImageUrl = (url) => {
  if (!url) return '';
  const filename = url.split('/').pop(); // 提取文件名部分
  return `http://localhost:8080/uploads/post/${filename}`;
};

const PostList = ({ search }) => {
  const [posts, setPosts] = useState([]);
  const [page, setPage] = useState(1);
  const [total, setTotal] = useState(0);
  const [loading, setLoading] = useState(true);
  const [errorMsg, setErrorMsg] = useState('');

  const size = 10;
  const totalPages = Math.ceil(total / size);

  useEffect(() => {
    const fetchPosts = async () => {
      setLoading(true);
      setErrorMsg('');
      try {
        const res = await axios.get(`/echo/forum/posts/getAllPosts`, {
          params: { page, pageSize: size, sortBy: 'createdAt', order: 'desc' }
        });

        if (!res.data || !Array.isArray(res.data.posts)) throw new Error('API返回格式不正确');

        const postsData = res.data.posts;

        const filteredPosts = postsData.filter(post =>
          post.title?.toLowerCase().includes(search.toLowerCase())
        );

        const postsWithProfiles = filteredPosts.map(post => ({
          ...post,
          username: post.username || '未知用户',
          avatarUrl: formatAvatarUrl(post.avatarUrl || ''),
          liked: false,
          collected: false,
          commentCount: 0
        }));

        setPosts(postsWithProfiles);
        setTotal(res.data.total || 0);
      } catch (err) {
        console.error('加载失败:', err);
        setErrorMsg('加载失败，请稍后重试');
      } finally {
        setLoading(false);
      }
    };

    fetchPosts();
  }, [page, search]);

  const toggleLike = async (postNo, liked, userId) => {
    try {
      if (liked) {
        await axios.post(`/echo/forum/posts/${postNo}/unlike`, { user_id: userId });
      } else {
        await likePost(postNo, userId);
      }

      setPosts(posts =>
        posts.map(post =>
          post.postNo === postNo
            ? {
                ...post,
                liked: !liked,
                likeCount: liked ? post.likeCount - 1 : post.likeCount + 1
              }
            : post
        )
      );
    } catch (err) {
      console.error('点赞失败:', err);
      alert('点赞操作失败，请稍后重试');
    }
  };

  const toggleCollect = async (postNo, collected, userId) => {
    try {
      if (collected) {
        await axios.delete(`/echo/forum/posts/${postNo}/uncollect`, {
          data: { user_id: userId }
        });
      } else {
        await axios.post(`/echo/forum/posts/${postNo}/collect`, {
          user_id: userId
        });
      }

      setPosts(posts =>
        posts.map(post =>
          post.postNo === postNo
            ? {
                ...post,
                collected: !collected,
                collectCount: collected ? post.collectCount - 1 : post.collectCount + 1
              }
            : post
        )
      );
    } catch (err) {
      console.error('收藏操作失败:', err);
      alert('收藏操作失败，请稍后重试');
    }
  };

  const handleDeletePost = async (postNo) => {
    if (window.confirm('确定要删除这篇帖子吗？')) {
      try {
        await axios.delete(`/echo/forum/posts/${postNo}/deletePost`);
        setPosts(posts => posts.filter(post => post.postNo !== postNo));
        if (posts.length === 1 && page > 1) setPage(page - 1);
      } catch (err) {
        console.error('删除帖子失败:', err);
        alert('删除帖子失败，请稍后再试');
      }
    }
  };

  return (
    <div className="post-list">
      {loading ? <p>加载中...</p> :
        errorMsg ? <p className="error-text">{errorMsg}</p> :
          posts.length === 0 ? <p>暂无帖子。</p> :
            posts.map(post => {
              const createdAtDate = new Date(post.createdAt);
              const timeText = isNaN(createdAtDate.getTime()) ? '时间未设置' : createdAtDate.toLocaleString();

              let coverImage = null;
              if (post.imgUrl) {
                const imgs = post.imgUrl.split(',').map(i => i.trim()).filter(Boolean);
                coverImage = imgs.length > 0 ? formatImageUrl(imgs[0]) : null;
              }

              return (
                <div key={post.postNo} className="post-card" style={{ backgroundColor: '#e9ded2' }}>
                  <div className="user-info">
                    <img
                      className="avatar"
                      src={post.avatarUrl}
                      alt="头像"
                    />
                    <span className="nickname" style={{ color: '#755e50' }}>{post.username}</span>
                  </div>

                  {coverImage && <img className="cover-image" src={coverImage} alt="封面" />}

                  <h3 style={{ color: '#000000' }}>{post.title || '无标题'}</h3>
                  <div className="post-meta">
                    <span>发布时间：{timeText}</span>
                    <div className="post-actions">
                      <button className="icon-btn" onClick={() => toggleLike(post.postNo, post.liked, post.user_id)}>
                        <GoodTwo theme="outline" size="24" fill={post.liked ? '#f00' : '#fff'} />
                        <span>{post.likeCount}</span>
                      </button>
                      <button className="icon-btn" onClick={() => toggleCollect(post.postNo, post.collected, post.user_id)}>
                        <Star theme="outline" size="24" fill={post.collected ? '#ffd700' : '#fff'} />
                        <span>{post.collectCount}</span>
                      </button>
                      <button className="icon-btn" onClick={() => handleDeletePost(post.postNo)}>
                        <Delete theme="outline" size="24" fill="#333" />
                      </button>
                    </div>
                  </div>
                  <div className="detail-button-wrapper">
                    <Link href={`/forum/post/${post.postNo}`} className="detail-button">查看详情</Link>
                  </div>
                </div>
              );
            })
      }

      <div className="pagination">
        <button disabled={page === 1} onClick={() => setPage(page - 1)}>上一页</button>
        <span>第 {page} 页 / 共 {totalPages} 页</span>
        <button disabled={page === totalPages} onClick={() => setPage(page + 1)}>下一页</button>
      </div>
    </div>
  );
};

export default PostList;
