import { createContext, useContext, useState, useCallback } from 'react';
import { fetchGroups, joinGroup, createPost } from '../services/groupService';

const GroupContext = createContext();

export const GroupProvider = ({ children }) => {
  const [groups, setGroups] = useState([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  const [category, setCategory] = useState('');
  const [name, setName] = useState('');
  const [page, setPage] = useState(1);
  const [size, setSize] = useState(10);
  const [totalPages, setTotalPages] = useState(1);
  const [sortBy, setSortBy] = useState('member_count');
  const [joinStatus, setJoinStatus] = useState({});

  // 获取小组列表
  const fetchGroupList = useCallback(async () => {
    try {
      setLoading(true);
      setError(null);
      
      const requestData = {
        page,
        size,
        sort_by: sortBy
      };
      
      if (category) requestData.category = category;
      if (name) requestData.name = name;
      
      const response = await fetchGroups(requestData);
      
      if (response.status === 'success') {
        setGroups(response.items || []);
        setTotalPages(response.total_pages || 1);
      } else {
        setError('获取兴趣小组列表失败');
      }
    } catch (err) {
      setError('请求失败，请稍后再试');
    } finally {
      setLoading(false);
    }
  }, [category, name, page, size, sortBy]);

  // 加入小组
  const handleJoinGroup = useCallback(async (groupId, userId) => {
    if (!userId) return;

    try {
      setJoinStatus(prev => ({ ...prev, [groupId]: '请求中' }));
      const response = await joinGroup(groupId, userId);
      
      if (response.status === 'success') {
        setJoinStatus(prev => ({ ...prev, [groupId]: '加入成功' }));
        fetchGroupList(); // 刷新列表
      } else {
        setJoinStatus(prev => ({ ...prev, [groupId]: '加入失败' }));
      }
    } catch (error) {
      setJoinStatus(prev => ({ ...prev, [groupId]: '请求失败' }));
    }
  }, [fetchGroupList]);

  // 创建帖子
  const handleCreatePost = useCallback(async (groupId, userId, content, title, images) => {
    if (!userId || !content || !title) return;

    try {
      const response = await createPost(groupId, userId, content, title, images);
      
      if (response.post_id) {
        return true; // 成功
      } else {
        throw new Error('创建帖子失败');
      }
    } catch (error) {
      console.error('创建帖子错误:', error);
      return false;
    }
  }, []);

  const value = {
    groups,
    loading,
    error,
    category,
    setCategory,
    name,
    setName,
    page,
    setPage,
    size,
    setSize,
    totalPages,
    sortBy,
    setSortBy,
    joinStatus,
    fetchGroupList,
    handleJoinGroup,
    handleCreatePost
  };

  return (
    <GroupContext.Provider value={value}>
      {children}
    </GroupContext.Provider>
  );
};

export const useGroupStore = () => useContext(GroupContext);