import React, { createContext, useContext, useState, useEffect } from 'react';

const avatarBaseUrl = process.env.REACT_APP_AVATAR_BASE_URL || '';

export const UserContext = createContext();

export const UserProvider = ({ children }) => {
  const [user, setUser] = useState(null);
  const [loading, setLoading] = useState(true);

  // ✅ 标准化用户数据结构
  const formatUser = (raw) => {
    if (!raw) return null;
    return {
      userId: raw.userId || raw.id || null,
      username: raw.username || '匿名用户',
      avatarUrl: raw.avatarUrl ? `${avatarBaseUrl}${raw.avatarUrl}` : null,
      ...raw, // 保留其它字段（如 email, level 等）
    };
  };

  useEffect(() => {
    const storedUser = localStorage.getItem('user');
    if (storedUser) {
      try {
        const parsed = JSON.parse(storedUser);
        setUser(formatUser(parsed));
      } catch (err) {
        console.error('本地用户信息解析失败:', err);
        localStorage.removeItem('user');
        setUser(null);
      }
    } else {
      // 设置默认用户
      const defaultUser = {
        userId: 1,
        username: '测试用户',
        avatarUrl: null,
      };
      localStorage.setItem('user', JSON.stringify(defaultUser));
      setUser(formatUser(defaultUser));
    }
    setLoading(false);
  }, []);

  const saveUser = (userData) => {
    if (userData && (userData.userId || userData.id)) {
      const formatted = formatUser(userData);
      localStorage.setItem('user', JSON.stringify(userData)); // 原始数据存储
      setUser(formatted); // 格式化后使用
    } else {
      console.error('无效的用户数据:', userData);
    }
  };

  const logout = () => {
    localStorage.removeItem('user');
    setUser(null);
  };

  return (
    <UserContext.Provider value={{ user, saveUser, logout, loading }}>
      {children}
    </UserContext.Provider>
  );
};

export const useUser = () => {
  const context = useContext(UserContext);
  if (!context) {
    throw new Error('useUser must be used within a UserProvider');
  }
  return context;
};
