import React, { useEffect, useState } from 'react';
import axios from 'axios';
import { Link } from 'wouter';
import { GoodTwo, Star, Delete } from '@icon-park/react';
import { likePost } from '../../posts-detail/api';
import './PostList.css';

const PostList = ({ search }) => {
  const [posts, setPosts] = useState([]);
  const [page, setPage] = useState(1);
  const [total, setTotal] = useState(0);
  const [loading, setLoading] = useState(true);
  const [errorMsg, setErrorMsg] = useState('');

  const size = 10;
  const totalPages = Math.ceil(total / size);

  useEffect(() => {
    const fetchPosts = async () => {
      setLoading(true);
      setErrorMsg('');
      try {
        const res = await axios.get(`/echo/forum/posts/getAllPosts`, {
          params: {
            page,
            pageSize: size,
            sortBy: 'createdAt',
            order: 'desc'
          }
        });

        // 检查响应结构是否符合预期
        if (!res.data || !Array.isArray(res.data.posts)) {
          throw new Error('API返回格式不正确');
        }

        const postsData = res.data.posts || [];

        const userIds = [...new Set(postsData.map(post => post.user_id))];

        const profiles = await Promise.all(userIds.map(async id => {
          try {
            const r = await axios.get(`/echo/user/profile`, {
              params: { user_id: id }
            });
            return { id, profile: r.data };
          } catch (e) {
            return { id, profile: { nickname: '未知用户', avatar_url: 'default-avatar.png' } };
          }
        }));

        const userMap = {};
        profiles.forEach(({ id, profile }) => {
          userMap[id] = profile;
        });

        const postsWithProfiles = postsData
          .filter(post => post.title.toLowerCase().includes(search.toLowerCase()))
          .map(post => ({
            ...post,
            userProfile: userMap[post.user_id] || { nickname: '未知用户', avatar_url: 'default-avatar.png' },
            liked: false,
            collected: false,
            commentCount: 0
          }));

        setPosts(postsWithProfiles);
        setTotal(res.data.total || 0);
      } catch (err) {
        console.error('加载失败:', err);
        setErrorMsg('加载失败，请稍后重试');
      } finally {
        setLoading(false);
      }
    };

    fetchPosts();
  }, [page, search]);

  const toggleLike = async (postNo, liked, userId) => {
  try {
    if (liked) {
      // 修改为 POST 请求，并带上 user_id 参数
      await axios.post(`/echo/forum/posts/${postNo}/unlike`, {
        user_id: userId
      });
    } else {
      await likePost(postNo, userId); // 你已有的点赞逻辑
    }

    setPosts(posts =>
      posts.map(post =>
        post.postNo === postNo
          ? { ...post, liked: !liked, likeCount: liked ? post.likeCount - 1 : post.likeCount + 1 }
          : post
      )
    );
  } catch (err) {
    console.error('点赞失败:', err);
    alert('点赞操作失败，请稍后重试');
  }
};


  // 收藏帖子
  const toggleCollect = async (postNo, collected, userId) => {
  try {
    if (collected) {
      // 取消收藏：DELETE 请求 + JSON 请求体
      await axios.delete(`/echo/forum/posts/${postNo}/uncollect`, {
        data: { user_id: userId } // 注意：DELETE 请求的请求体需放在 data 字段中
      });
    } else {
      // 收藏：POST 请求 + JSON 请求体
      await axios.post(`/echo/forum/posts/${postNo}/collect`, {
        user_id: userId
      });
    }

    setPosts(posts =>
      posts.map(post =>
        post.postNo === postNo
          ? {
              ...post,
              collected: !collected,
              collectCount: collected ? post.collectCount - 1 : post.collectCount + 1
            }
          : post
      )
    );
  } catch (err) {
    console.error('收藏操作失败:', err.response?.data || err.message);
    alert('收藏操作失败，请稍后重试');
  }
};



  // 删除帖子
  const handleDeletePost = async (postNo) => {
    if (window.confirm('确定要删除这篇帖子吗？')) {
      try {
        await axios.delete(`/echo/forum/posts/${postNo}/deletePost`);
        
        // 从列表中移除已删除的帖子
        setPosts(posts => posts.filter(post => post.postNo !== postNo));
        
        // 如果删除后当前页没有帖子了，尝试加载上一页
        if (posts.length === 1 && page > 1) {
          setPage(page - 1);
        }
      } catch (err) {
        console.error('删除帖子失败:', err);
        alert('删除帖子失败，请稍后再试');
      }
    }
  };

  return (
    <div className="post-list">
      {loading ? <p>加载中...</p> :
        errorMsg ? <p className="error-text">{errorMsg}</p> :
          posts.length === 0 ? <p>暂无帖子。</p> :
            posts.map(post => (
              <div
                key={post.postNo}
                className="post-card"
                style={{ backgroundColor: '#e9ded2' }}
              >
                <div className="user-info">
                  <img className="avatar" src={post.userProfile.avatar_url} alt="头像" />
                  <span className="nickname" style={{ color: '#755e50' }}>{post.userProfile.nickname}</span>
                </div>
                {post.imgUrl && (
                  <img className="cover-image" src={post.imgUrl} alt="封面" />
                )}
                <h3 style={{ color: '#000000' }}>{post.title}</h3>
                <div className="post-meta">
                  <span>发布时间：{new Date(post.createdAt).toLocaleString()}</span>
                  <div className="post-actions">
                    <button className="icon-btn" onClick={() => toggleLike(post.postNo, post.liked, post.user_id)}>
                      <GoodTwo theme="outline" size="24" fill={post.liked ? '#f00' : '#fff'} />
                      <span>{post.likeCount}</span>
                    </button>

                    <button className="icon-btn" onClick={() => toggleCollect(post.postNo, post.collected, post.user_id)}>
                      <Star theme="outline" size="24" fill={post.collected ? '#ffd700' : '#fff'} />
                      <span>{post.collectCount}</span>
                    </button>
                    
                    <button className="icon-btn" onClick={() => handleDeletePost(post.postNo)}>
                      <Delete theme="outline" size="24" fill="#333" />
                    </button>
                  </div>
                </div>
                <div className="detail-button-wrapper">
                  <Link href={`/forum/post/${post.postNo}`} className="detail-button">查看详情</Link>
                </div>
              </div>
            ))
      }

      <div className="pagination">
        <button disabled={page === 1} onClick={() => setPage(page - 1)}>上一页</button>
        <span>第 {page} 页 / 共 {totalPages} 页</span>
        <button disabled={page === totalPages} onClick={() => setPage(page + 1)}>下一页</button>
      </div>
    </div>
  );
};

export default PostList;