package com.pt5.pthouduan.controller;

import com.pt5.pthouduan.entity.Post;
import com.pt5.pthouduan.service.PostService;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.springframework.web.multipart.MultipartFile;

import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.Mockito.*;

class PostControllerTest {

    @Mock
    private PostService postService;

    @InjectMocks
    private PostController postController;

    @BeforeEach
    void setUp() {
        MockitoAnnotations.openMocks(this);
    }

    @Test
    void createPost_WithValidData_ShouldReturnTrue() {
        MultipartFile photo = mock(MultipartFile.class);
        when(photo.isEmpty()).thenReturn(true); // 模拟没有上传图片

        Post mockPost = new Post();
        mockPost.setPostid(1);
        when(postService.createPost(any(Post.class))).thenReturn(mockPost);

        boolean result = postController.createPost(
                1L, "标题", "内容", "标签", "全站", true, photo
        );

        assertTrue(result);
    }

    @Test
    void deletePost_ShouldReturnTrue_WhenServiceSucceeds() {
        when(postService.deletePost(1)).thenReturn(true);
        boolean result = postController.deletePost(1);
        assertTrue(result);
    }

    @Test
    void updatePost_ShouldReturnTrue_WhenUpdateSucceeds() {
        Post post = new Post();
        post.setPostid(1);
        when(postService.updatePost(post)).thenReturn(true);
        assertTrue(postController.updatePost(post));
    }

    @Test
    void searchPosts_ShouldReturnMatchedPosts() {
        Post post1 = new Post();
        post1.setPostTitle("关键字匹配");

        when(postService.searchPostsByKeyword("关键字")).thenReturn(List.of(post1));

        List<Post> result = postController.searchPosts("关键字");

        assertEquals(1, result.size());
        assertEquals("关键字匹配", result.get(0).getPostTitle());
    }

    @Test
    void likePost_ShouldReturnTrue() {
        when(postService.incrementLikes(1)).thenReturn(true);
        assertTrue(postController.likePost(1));
    }

    @Test
    void unlikePost_ShouldReturnTrue() {
        when(postService.decrementLikes(1)).thenReturn(true);
        assertTrue(postController.unlikePost(1));
    }

    @Test
    void pinPost_ShouldReturnTrue() {
        when(postService.setPinnedStatus(1, true)).thenReturn(true);
        assertTrue(postController.pinPost(1));
    }

    @Test
    void unpinPost_ShouldReturnTrue() {
        when(postService.setPinnedStatus(1, false)).thenReturn(true);
        assertTrue(postController.unpinPost(1));
    }

    @Test
    void findByUserid_ShouldReturnUserPosts() {
        Post post = new Post();
        post.setUserid(1L);
        when(postService.findByUserid(1L)).thenReturn(List.of(post));

        List<Post> result = postController.findByUserid(1L);
        assertEquals(1, result.size());
        assertEquals(1L, result.get(0).getUserid());
    }

    @Test
    void findPinnedPosts_ShouldReturnPinnedList() {
        Post post = new Post();
        post.setIsPinned(true);
        when(postService.findPinnedPosts()).thenReturn(List.of(post));

        List<Post> result = postController.findPinnedPosts();
        assertEquals(1, result.size());
        assertTrue(result.get(0).getIsPinned());
    }

    @Test
    void togglePinStatus_ShouldToggleAndReturnNewStatus() {
        Post post = new Post();
        post.setPostid(1);
        post.setIsPinned(false);

        when(postService.getById(1)).thenReturn(post);
        when(postService.setPinnedStatus(1, true)).thenReturn(true);

        Boolean result = postController.togglePinStatus(1);
        assertTrue(result);
    }

    @Test
    void getAllPostsSorted_ShouldReturnAllPosts() {
        Post post1 = new Post();
        Post post2 = new Post();
        when(postService.getAllPostsSorted()).thenReturn(Arrays.asList(post1, post2));

        List<Post> posts = postController.getAllPostsSorted();
        assertEquals(2, posts.size());
    }
}
