package com.pt5.pthouduan.ControllerTest;


import com.pt5.pthouduan.controller.InvitesController;
import com.pt5.pthouduan.service.InviteService;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import java.util.HashMap;
import java.util.Map;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.Mockito.*;

class InvitesControllerTest {

    @Mock
    private InviteService inviteService;

    @InjectMocks
    private InvitesController invitesController;

    @BeforeEach
    void setUp() {
        MockitoAnnotations.openMocks(this);
    }

    @Test
    void getUserInfo_Success() {
        // 准备测试数据
        Map<String, Object> expectedResponse = new HashMap<>();
        expectedResponse.put("success", true);
        expectedResponse.put("invites", 5);
        expectedResponse.put("username", "testuser");

        // 模拟服务层行为
        when(inviteService.getInvitesByUsername(anyString()))
                .thenReturn(expectedResponse);

        // 执行测试
        Map<String, Object> actualResponse = invitesController.getUserInfo("testuser");

        // 验证结果
        assertEquals(expectedResponse, actualResponse);
        verify(inviteService, times(1)).getInvitesByUsername("testuser");
    }

    @Test
    void getUserInfo_UserNotFound() {
        Map<String, Object> expectedResponse = new HashMap<>();
        expectedResponse.put("success", false);
        expectedResponse.put("message", "用户不存在");

        when(inviteService.getInvitesByUsername(anyString()))
                .thenReturn(expectedResponse);

        Map<String, Object> actualResponse = invitesController.getUserInfo("nonexistent");

        assertEquals(expectedResponse, actualResponse);
        assertFalse((Boolean) actualResponse.get("success"));
    }

    @Test
    void getUserInfo_EmptyUsername() {
        Map<String, Object> expectedResponse = new HashMap<>();
        expectedResponse.put("success", false);
        expectedResponse.put("message", "用户名不能为空");

        when(inviteService.getInvitesByUsername(""))
                .thenReturn(expectedResponse);

        Map<String, Object> actualResponse = invitesController.getUserInfo("");

        assertEquals(expectedResponse, actualResponse);
        assertEquals("用户名不能为空", actualResponse.get("message"));
    }

    @Test
    void getUserInfo_ServiceException() {
        when(inviteService.getInvitesByUsername(anyString()))
                .thenThrow(new RuntimeException("数据库连接失败"));

        assertThrows(RuntimeException.class, () -> {
            invitesController.getUserInfo("testuser");
        });
    }

    // 边界条件测试
    @Test
    void getUserInfo_LongUsername() {
        String longUsername = "a".repeat(256);
        Map<String, Object> expectedResponse = new HashMap<>();
        expectedResponse.put("success", false);
        expectedResponse.put("message", "用户名过长");

        when(inviteService.getInvitesByUsername(longUsername))
                .thenReturn(expectedResponse);

        Map<String, Object> actualResponse = invitesController.getUserInfo(longUsername);

        assertEquals(expectedResponse, actualResponse);
    }

    // 性能测试示例（实际项目中可能需要单独的测试类）
    @Test
    void getUserInfo_Performance() {
        // 模拟快速响应
        when(inviteService.getInvitesByUsername(anyString()))
                .thenAnswer(invocation -> {
                    Map<String, Object> response = new HashMap<>();
                    response.put("success", true);
                    return response;
                });

        long startTime = System.currentTimeMillis();
        invitesController.getUserInfo("testuser");
        long duration = System.currentTimeMillis() - startTime;

        assertTrue(duration < 100, "响应时间应小于100毫秒");
    }
}
