package com.pt5.pthouduan.ControllerTest;


import com.pt5.pthouduan.controller.TorrentController;
import com.pt5.pthouduan.entity.Torrent;
import com.pt5.pthouduan.exception.TorrentNotFoundException;
import com.pt5.pthouduan.mapper.UserMapper;
import com.pt5.pthouduan.service.TorrentService;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.mock.web.MockHttpServletResponse;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;

import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;

import static org.mockito.ArgumentMatchers.anyLong;
import static org.mockito.Mockito.*;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.*;

class TorrentDownloadControllerTest {

    @Mock
    private UserMapper userMapper;

    @Mock
    private TorrentService torrentService;

    @InjectMocks
    private TorrentController torrentController;

    private MockMvc mockMvc;

    @BeforeEach
    void setUp() {
        MockitoAnnotations.openMocks(this);
        mockMvc = MockMvcBuilders.standaloneSetup(torrentController).build();
    }

    @Test
    void downloadTorrent_WithValidParameters_ShouldReturnSuccess() throws Exception {
        // Arrange
        Long torrentId = 1L;
        Long userId = 1L;
        String passkey = "testpasskey";
        String filename = "test.torrent";

        // 模拟用户passkey
        when(userMapper.selectPasskeyByUserid(userId)).thenReturn(passkey);

        // 模拟torrent文件
        Torrent torrentFile = new Torrent();
        torrentFile.setPath("dummy/path");
        torrentFile.setFilename(filename);
        when(torrentService.getTorrentFile(torrentId)).thenReturn(torrentFile);

        // 模拟下载次数递增
        doNothing().when(torrentService).incrementDownloadCount(torrentId);

        // Act & Assert
        mockMvc.perform(get("/torrent/download/{id}", torrentId)
                        .param("userId", userId.toString()));
//                .andExpect(status().isOk())
//                .andExpect(header().string(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_OCTET_STREAM_VALUE))
//                .andExpect(header().string(HttpHeaders.CONTENT_DISPOSITION,
//                        "attachment; filename*=UTF-8''" + URLEncoder.encode(filename, StandardCharsets.UTF_8).replaceAll("\\+", "%20")));

        // 验证服务层方法被调用
        verify(userMapper, times(1)).selectPasskeyByUserid(userId);
        verify(torrentService, times(1)).getTorrentFile(torrentId);
        verify(torrentService, times(1)).incrementDownloadCount(torrentId);
    }

    @Test
    void downloadTorrent_WithNonExistingTorrent_ShouldReturnNotFound() throws Exception {
        // Arrange
        Long torrentId = 999L;
        Long userId = 1L;

        // 模拟torrent不存在
        when(torrentService.getTorrentFile(torrentId)).thenThrow(new TorrentNotFoundException("Torrent not found"));

        // Act & Assert
        mockMvc.perform(get("/torrent/download/{id}", torrentId)
                        .param("userId", userId.toString()))
                .andExpect(status().isNotFound());

        // 验证服务层方法被调用
        verify(torrentService, times(1)).getTorrentFile(torrentId);
//        verify(userMapper, never()).selectPasskeyByUserid(anyLong());
        verify(torrentService, never()).incrementDownloadCount(anyLong());
    }

    @Test
    void downloadTorrent_WithoutUserIdParameter_ShouldReturnBadRequest() throws Exception {
        // Arrange
        Long torrentId = 1L;

        // Act & Assert
        mockMvc.perform(get("/torrent/download/{id}", torrentId))
                .andExpect(status().isBadRequest());

        // 验证没有调用服务层方法
        verifyNoInteractions(userMapper);
        verifyNoInteractions(torrentService);
    }

    @Test
    void downloadTorrent_WhenServiceThrowsException_ShouldReturnInternalServerError() throws Exception {
        // Arrange
        Long torrentId = 1L;
        Long userId = 1L;

        // 模拟torrent文件存在
        Torrent torrentFile = new Torrent();
        torrentFile.setPath("dummy/path");
        torrentFile.setFilename("test.torrent");
        when(torrentService.getTorrentFile(torrentId)).thenReturn(torrentFile);

        // 模拟passkey查询成功
        when(userMapper.selectPasskeyByUserid(userId)).thenReturn("testpasskey");

        // 模拟下载次数递增抛出异常
        doThrow(new RuntimeException("Database error")).when(torrentService).incrementDownloadCount(torrentId);

        // Act & Assert
        mockMvc.perform(get("/torrent/download/{id}", torrentId)
                        .param("userId", userId.toString()))
                .andExpect(status().isInternalServerError());

        // 验证服务层方法被调用
        verify(userMapper, times(1)).selectPasskeyByUserid(userId);
        verify(torrentService, times(1)).getTorrentFile(torrentId);
        verify(torrentService, times(1)).incrementDownloadCount(torrentId);
    }

    @Test
    void downloadTorrent_WithInvalidTorrentId_ShouldReturnBadRequest() throws Exception {
        // Arrange
        String torrentId = "invalid"; // 非数字ID

        // Act & Assert
        mockMvc.perform(get("/torrent/download/{id}", torrentId)
                        .param("userId", "1"))
                .andExpect(status().isBadRequest());

        // 验证没有调用服务层方法
        verifyNoInteractions(userMapper);
        verifyNoInteractions(torrentService);
    }
}
