package com.pt5.pthouduan.controller;

import com.pt5.pthouduan.entity.Post;
import com.pt5.pthouduan.service.PostService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.io.File;
import java.io.IOException;
import java.time.LocalDateTime;
import java.util.List;
import java.util.UUID;

/**
 * <p>
 *  帖子控制器
 * </p>
 *
 * 功能：创建帖子（支持上传图片、置顶、范围控制）、点赞、取消点赞、关键词搜索、删除、更新、置顶与取消置顶
 *
 * @author
 * @since 2025-05-10
 */
@CrossOrigin(origins = {"http://localhost:5173", "http://localhost:3000"})
@Controller
@RequestMapping("/post")
public class PostController {

    @Autowired
    private PostService postService;

    // 创建帖子（支持图片上传）
    @PostMapping("/create")
    @ResponseBody
    public boolean createPost(
            @RequestParam("userid") Long userid,
            @RequestParam("post_title") String post_title,
            @RequestParam("post_content") String post_content,
            @RequestParam(value = "tags", required = false) String tags,
            @RequestParam(value = "rannge", required = false) String rannge,
            @RequestParam(value = "is_pinned", required = false) Boolean is_pinned,
            @RequestParam(value = "photo", required = false) MultipartFile photoFile
    ) {
        Post post = new Post();
        post.setUserid(userid);
        post.setPostTitle(post_title);
        post.setPostContent(post_content);
        post.setTags(tags);
        post.setRannge(rannge);
        post.setIsPinned(is_pinned != null && is_pinned);
        post.setPostCreatedTime(LocalDateTime.now());
        post.setUpdatedTime(LocalDateTime.now());
        post.setLikes(0);


        // 保存图片
        if (photoFile != null && !photoFile.isEmpty()) {
            String uploadDir = "D:/postuploads/";
            File dir = new File(uploadDir);
            if (!dir.exists()) dir.mkdirs();

            String fileName = UUID.randomUUID() + "_" + photoFile.getOriginalFilename();
            File dest = new File(uploadDir + fileName);
            try {
                photoFile.transferTo(dest);
                post.setPhoto("/images/" + fileName);
            } catch (IOException e) {
                e.printStackTrace();
                return false;
            }
        }

        return postService.createPost(post) != null;
    }



    // 删除帖子
    @DeleteMapping("/delete/{postid}")
    @ResponseBody
    public boolean deletePost(@PathVariable Integer postid) {
        return postService.deletePost(postid);
    }

    // 更新帖子（包括置顶、范围、照片等）
    @PutMapping("/update")
    @ResponseBody
    public boolean updatePost(@RequestBody Post post) {
        return postService.updatePost(post);
    }

    // 关键词搜索
    @GetMapping("/search")
    @ResponseBody
    public List<Post> searchPosts(@RequestParam String keyword) {
        return postService.searchPostsByKeyword(keyword);
    }

    // 点赞
    @PutMapping("/like/{postid}")
    @ResponseBody
    public boolean likePost(@PathVariable Integer postid) {
        return postService.incrementLikes(postid);
    }

    @GetMapping("/all")
    @ResponseBody
    public List<Post> getAllPostsSorted() {
        return postService.getAllPostsSorted();
    }


    // 取消点赞
    @PutMapping("/unlike/{postid}")
    @ResponseBody
    public boolean unlikePost(@PathVariable Integer postid) {
        return postService.decrementLikes(postid);
    }

    // 置顶帖子
    @PutMapping("/pin/{postid}")
    @ResponseBody
    public boolean pinPost(@PathVariable Integer postid) {
        return postService.setPinnedStatus(postid, true);
    }

    // 取消置顶
    @PutMapping("/unpin/{postid}")
    @ResponseBody
    public boolean unpinPost(@PathVariable Integer postid) {
        return postService.setPinnedStatus(postid, false);
    }

    // 根据用户ID获取该用户所有帖子
    @GetMapping("/findByUserid")
    @ResponseBody
    public List<Post> findByUserid(@RequestParam Long userid) {
        return postService.findByUserid(userid);
    }

    // 根据是否置顶查找帖子
    @GetMapping("/findPinned")
    @ResponseBody
    public List<Post> findPinnedPosts() {
        return postService.findPinnedPosts();
    }

   

    // 添加切换置顶状态的新端点
    @PutMapping("/togglePin/{postid}")
    @ResponseBody
    public Boolean togglePinStatus(@PathVariable Integer postid) {
        Post post = postService.getById(postid);
        if (post == null) return null;

        boolean newStatus = !post.getIsPinned();
        boolean result = postService.setPinnedStatus(postid, newStatus);
        return result ? newStatus : null; // ✅ 返回新状态
    }
}
