package com.pt5.pthouduan.controller;

import com.pt5.pthouduan.entity.Activity;
import com.pt5.pthouduan.service.ActivityService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.io.File;
import java.io.IOException;
import java.time.LocalDateTime;
import java.util.List;
import java.util.UUID;
import java.util.stream.Collectors;

/**
 * <p>
 * 活动控制器
 * </p>
 *
 * 功能：增删改查活动公告（支持上传图片）
 *
 * @author ym
 * @since 2025-05-16
 */
@CrossOrigin(origins = {"http://localhost:5173", "http://localhost:3000"})
@Controller
@RequestMapping("/activity")
public class ActivityController {

    @Autowired
    private ActivityService activityService;

    // 获取所有 is_show == 0 的活动标题和图片（活动预览）
    @GetMapping("/preview")
    @ResponseBody
    public List<Activity> getVisibleActivityPreviews() {
        List<Activity> allVisible = activityService.findByIsShow(0);
        return allVisible.stream().map(activity -> {
            Activity preview = new Activity();
            preview.setActivityid(activity.getActivityid());
            preview.setContent(activity.getContent());
            preview.setTitle(activity.getTitle());
            preview.setAward(activity.getAward());
            preview.setTime(activity.getTime());
            preview.setPhoto(activity.getPhoto());
            return preview;
        }).collect(Collectors.toList());
    }

    // 获取所有 is_show == 0 的完整活动信息
    @GetMapping("/full")
    @ResponseBody
    public List<Activity> getVisibleActivities() {
        return activityService.findByIsShow(0);
    }

    // 创建新的公告（支持图片上传）
    @PostMapping("/create")
    @ResponseBody
    public boolean createActivity(
            @RequestParam("title") String title,
            @RequestParam("content") String content,
            @RequestParam(value = "photo", required = false) MultipartFile photoFile,
            @RequestParam(value = "isShow", required = false, defaultValue = "0") Integer isShow,
            @RequestParam(value = "award", required = false, defaultValue = "0") Integer award
    ) {
        Activity activity = new Activity();
        activity.setTitle(title);
        activity.setContent(content);
        activity.setIs_show(isShow);
        activity.setAward(award);
        activity.setTime(LocalDateTime.now()); // 设置当前时间

        // 处理图片上传
        if (photoFile != null && !photoFile.isEmpty()) {
            String uploadDir = "D:/activityuploads/";
            File dir = new File(uploadDir);
            if (!dir.exists()) dir.mkdirs();

            String fileName = UUID.randomUUID() + "_" + photoFile.getOriginalFilename();
            File dest = new File(uploadDir + fileName);
            try {
                photoFile.transferTo(dest);
                // 设置图片访问路径（请确保 /images 映射到了 uploadDir）
                activity.setPhoto("/activity/" + fileName);
            } catch (IOException e) {
                e.printStackTrace();
                return false;
            }
        }

        return activityService.save(activity);
    }


    // 删除公告（根据ID）
    @DeleteMapping("/delete/{id}")
    @ResponseBody
    public boolean deleteActivity(@PathVariable Integer id) {
        return activityService.removeById(id);
    }

    // 获取所有活动（无论展示状态）
    @GetMapping("/all")
    @ResponseBody
    public List<Activity> getAllActivities() {
        // 假设 activityService 有 list() 方法，返回所有活动
        return activityService.list();
    }

    //根据题目搜索公告
    @GetMapping("/search")
    @ResponseBody
    public List<Activity> searchActivitiesByTitle(@RequestParam("title") String title) {
        return activityService.searchByTitle(title);
    }

    // 修改公告（根据ID）
    @PutMapping("/update")
    @ResponseBody
    public boolean updateActivity(
            @RequestParam("activityid") Integer id,
            @RequestParam("title") String title,
            @RequestParam("content") String content,
            @RequestParam(value = "photo", required = false) MultipartFile photoFile,
            @RequestParam(value = "isShow", required = false, defaultValue = "0") Integer isShow,
            @RequestParam(value = "award", required = false, defaultValue = "0") Integer award
    ) {
        Activity activity = activityService.getById(id);
        if (activity == null) {
            return false;
        }

        activity.setTitle(title);
        activity.setContent(content);
        activity.setIs_show(isShow);
        activity.setAward(award);

        // 上传新图片（可选）
        if (photoFile != null && !photoFile.isEmpty()) {
            String uploadDir = "D:/activityuploads/";
            File dir = new File(uploadDir);
            if (!dir.exists()) dir.mkdirs();

            String fileName = UUID.randomUUID() + "_" + photoFile.getOriginalFilename();
            File dest = new File(uploadDir + fileName);
            try {
                photoFile.transferTo(dest);
                activity.setPhoto("/activity/" + fileName);
            } catch (IOException e) {
                e.printStackTrace();
                return false;
            }
        }

        return activityService.updateById(activity);
    }
}
