package com.pt5.pthouduan.ControllerTest;

import com.pt5.pthouduan.controller.ShopController;
import com.pt5.pthouduan.service.ShopService;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.springframework.http.ResponseEntity;

import java.util.HashMap;
import java.util.Map;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.ArgumentMatchers.*;
import static org.mockito.Mockito.*;

class ShopControllerTest {

    @Mock
    private ShopService shopService;

    @InjectMocks
    private ShopController shopController;

    @BeforeEach
    void setUp() {
        MockitoAnnotations.openMocks(this);
    }

    @Test
    void soldDecoration_Success() {
        // 准备测试数据
        Map<String, Object> expectedResponse = new HashMap<>();
        expectedResponse.put("success", true);
        expectedResponse.put("message", "装饰购买成功");

        // 模拟服务层行为
        when(shopService.buyDecoration(anyString(), anyString(), anyInt()))
                .thenReturn(expectedResponse);

        // 执行测试
        Map<String, Object> actualResponse = shopController.soldDecoration("user1", "decoration1", 100);

        // 验证结果
        assertEquals(expectedResponse, actualResponse);
        verify(shopService, times(1)).buyDecoration("user1", "decoration1", 100);
    }

    @Test
    void soldDecoration_Failure() {
        Map<String, Object> expectedResponse = new HashMap<>();
        expectedResponse.put("success", false);
        expectedResponse.put("message", "余额不足");

        when(shopService.buyDecoration(anyString(), anyString(), anyInt()))
                .thenReturn(expectedResponse);

        Map<String, Object> actualResponse = shopController.soldDecoration("user1", "decoration1", 1000);

        assertEquals(expectedResponse, actualResponse);
        assertFalse((Boolean) actualResponse.get("success"));
    }

    @Test
    void soldUpload_Success() {
        Map<String, Object> expectedResponse = new HashMap<>();
        expectedResponse.put("success", true);
        expectedResponse.put("upload", 500);

        when(shopService.buyUpload(anyString(), anyInt(), anyInt()))
                .thenReturn(expectedResponse);

        Map<String, Object> actualResponse = shopController.soldUpload("user1", 50, 500);

        assertEquals(expectedResponse, actualResponse);
        verify(shopService).buyUpload("user1", 50, 500);
    }

    @Test
    void soldInvite_Success() {
        Map<String, Object> expectedResponse = new HashMap<>();
        expectedResponse.put("success", true);
        expectedResponse.put("inviteCode", "INV123");

        when(shopService.setbuyername(anyString(), anyInt()))
                .thenReturn(expectedResponse);

        Map<String, Object> actualResponse = shopController.soldInvite("user1", 200);

        assertEquals(expectedResponse, actualResponse);
        assertNotNull(actualResponse.get("inviteCode"));
    }

    @Test
    void soldInvite_InvalidPrice() {
        Map<String, Object> expectedResponse = new HashMap<>();
        expectedResponse.put("success", false);
        expectedResponse.put("message", "价格无效");

        when(shopService.setbuyername(anyString(), eq(0)))
                .thenReturn(expectedResponse);

        Map<String, Object> actualResponse = shopController.soldInvite("user1", 0);

        assertEquals(expectedResponse, actualResponse);
        assertFalse((Boolean) actualResponse.get("success"));
    }

    // 边界条件测试
    @Test
    void soldDecoration_ZeroPrice() {
        Map<String, Object> expectedResponse = new HashMap<>();
        expectedResponse.put("success", false);
        expectedResponse.put("message", "价格必须大于0");

        when(shopService.buyDecoration(anyString(), anyString(), eq(0)))
                .thenReturn(expectedResponse);

        Map<String, Object> actualResponse = shopController.soldDecoration("user1", "decoration1", 0);

        assertEquals(expectedResponse, actualResponse);
    }

    // 异常情况测试
    @Test
    void soldDecoration_ServiceThrowsException() {
        when(shopService.buyDecoration(anyString(), anyString(), anyInt()))
                .thenThrow(new RuntimeException("服务异常"));

        assertThrows(RuntimeException.class, () -> {
            shopController.soldDecoration("user1", "decoration1", 100);
        });
    }
}